/**
 * @file
 * Table Advanced UI plugin.
 *
 * Adds toolbar button and balloon form for editing table attributes.
 *
 * @module tableAdvanced/TableAdvancedUI
 */

import { Plugin } from 'ckeditor5/src/core';
import {
  ButtonView,
  ContextualBalloon,
  clickOutsideHandler,
} from 'ckeditor5/src/ui';
import TableAdvancedFormView from './ui/tableadvancedformview';
import tableIcon from '../theme/icons/table-advanced.svg';

/**
 * The Table Advanced UI plugin.
 *
 * Introduces the toolbar button and balloon panel.
 */
export default class TableAdvancedUI extends Plugin {
  /**
   * @inheritdoc
   */
  static get requires() {
    return [ContextualBalloon];
  }

  /**
   * @inheritdoc
   */
  static get pluginName() {
    return 'TableAdvancedUI';
  }

  /**
   * @inheritdoc
   */
  init() {
    const { editor } = this;

    this._balloon = editor.plugins.get(ContextualBalloon);
    this.formView = this._createFormView();

    this._createToolbarButton();
  }

  /**
   * Creates the toolbar button.
   *
   * @private
   */
  _createToolbarButton() {
    const { editor } = this;
    const { t } = editor.locale;

    editor.ui.componentFactory.add('tableAdvanced', (locale) => {
      const button = new ButtonView(locale);
      const command = editor.commands.get('setTableAttributes');

      button.set({
        label: t('Table Advanced'),
        icon: tableIcon,
        tooltip: true,
      });

      button.bind('isEnabled').to(command, 'isEnabled');

      this.listenTo(button, 'execute', () => {
        this._showUI();
      });

      return button;
    });
  }

  /**
   * Creates the form view.
   *
   * @return {module:tableAdvanced/ui/TableAdvancedFormView}
   *   The form view instance.
   * @private
   */
  _createFormView() {
    const { editor } = this;
    const config = editor.config.get('tableAdvanced') || {};

    const formView = new TableAdvancedFormView(editor.locale, config);

    // Execute command on submit.
    this.listenTo(formView, 'submit', () => {
      const values = formView.getValues();
      editor.execute('setTableAttributes', values);
      this._hideUI();
    });

    // Hide on cancel.
    this.listenTo(formView, 'cancel', () => {
      this._hideUI();
    });

    // Close on click outside.
    clickOutsideHandler({
      emitter: formView,
      activator: () => this._balloon.visibleView === formView,
      contextElements: [this._balloon.view.element],
      callback: () => this._hideUI(),
    });

    return formView;
  }

  /**
   * Shows the UI.
   *
   * @private
   */
  _showUI() {
    const { editor } = this;
    const command = editor.commands.get('setTableAttributes');

    if (!command.isEnabled) {
      return;
    }

    // Set current values.
    this.formView.setValues(command.value);

    this._balloon.add({
      view: this.formView,
      position: this._getBalloonPositionData(),
    });

    this.formView.focus();
  }

  /**
   * Hides the UI.
   *
   * @private
   */
  _hideUI() {
    this.formView.reset();
    this._balloon.remove(this.formView);
    this.editor.editing.view.focus();
  }

  /**
   * Gets balloon position data.
   *
   * @return {Object}
   *   The position configuration.
   * @private
   */
  _getBalloonPositionData() {
    const { editor } = this;
    const { editing } = editor;
    const { view } = editing;
    const { selection } = view.document;

    return {
      target: view.domConverter.viewRangeToDom(selection.getFirstRange()),
    };
  }
}
