<?php

declare(strict_types=1);

namespace Drupal\editor_advanced_table\Plugin\CKEditor5Plugin;

use Drupal\ckeditor5\Plugin\CKEditor5PluginConfigurableInterface;
use Drupal\ckeditor5\Plugin\CKEditor5PluginConfigurableTrait;
use Drupal\ckeditor5\Plugin\CKEditor5PluginDefault;
use Drupal\Core\Form\FormStateInterface;
use Drupal\editor\EditorInterface;

/**
 * CKEditor 5 Table Advanced plugin.
 *
 * Provides ID, Language Direction, and CSS class attributes for tables.
 *
 * @internal
 *   Plugin classes are internal.
 */
class TableAdvanced extends CKEditor5PluginDefault implements CKEditor5PluginConfigurableInterface {

  use CKEditor5PluginConfigurableTrait;

  /**
   * {@inheritdoc}
   *
   * @return array<string, bool>
   *   The default configuration.
   */
  public function defaultConfiguration(): array {
    return [
      'allow_classes' => TRUE,
      'allow_id' => TRUE,
      'allow_language_direction' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string, mixed>
   *   The modified form array.
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['info'] = [
      '#markup' => '<p>' . $this->t('Configure which advanced table attributes are available to editors.') . '</p>',
    ];

    $form['allow_classes'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('CSS classes (<code>class</code>)'),
      '#description' => $this->t('Allow editors to add CSS classes to tables (free-form text input).'),
      '#default_value' => $this->configuration['allow_classes'] ?? TRUE,
    ];

    $form['allow_id'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('ID (<code>id</code>)'),
      '#description' => $this->t('Allow editors to set an ID on tables (e.g., id="pricing-table").'),
      '#default_value' => $this->configuration['allow_id'] ?? TRUE,
    ];

    $form['allow_language_direction'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Language direction (<code>dir</code>)'),
      '#description' => $this->t('Allow editors to set text direction (LTR/RTL/Auto) on tables.'),
      '#default_value' => $this->configuration['allow_language_direction'] ?? TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // No validation needed.
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['allow_classes'] = (bool) $form_state->getValue('allow_classes');
    $this->configuration['allow_id'] = (bool) $form_state->getValue('allow_id');
    $this->configuration['allow_language_direction'] = (bool) $form_state->getValue('allow_language_direction');
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $static_plugin_config
   *   The static plugin configuration.
   * @param \Drupal\editor\EditorInterface $editor
   *   The editor instance.
   *
   * @return array<string, mixed>
   *   The dynamic plugin configuration.
   */
  public function getDynamicPluginConfig(array $static_plugin_config, EditorInterface $editor): array {
    // Return both our custom config and pass through the static
    // contentToolbar config.
    return [
      'tableAdvanced' => [
        'allowId' => (bool) $this->configuration['allow_id'],
        'allowLanguageDirection' => (bool) $this->configuration['allow_language_direction'],
        'allowClasses' => (bool) $this->configuration['allow_classes'],
      ],
      'table' => $static_plugin_config['table'] ?? [],
    ];
  }

}
