<?php

declare(strict_types=1);

namespace Drupal\Tests\editor_advanced_table\Kernel;

use Drupal\editor\Entity\Editor;
use Drupal\editor_advanced_table\Plugin\CKEditor5Plugin\TableAdvanced;
use Drupal\filter\Entity\FilterFormat;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests TableAdvanced plugin integration.
 *
 * @group editor_advanced_table
 */
class TableAdvancedIntegrationTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'ckeditor5',
    'editor',
    'editor_advanced_table',
    'filter',
    'system',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['filter', 'editor', 'ckeditor5']);
  }

  /**
   * Tests plugin is registered.
   */
  public function testPluginRegistration(): void {
    $manager = $this->container->get('plugin.manager.ckeditor5.plugin');
    $definitions = $manager->getDefinitions();

    $this->assertArrayHasKey('editor_advanced_table_table_advanced', $definitions);
  }

  /**
   * Tests plugin configuration.
   */
  public function testPluginConfiguration(): void {
    // Create filter format.
    $filter_format = FilterFormat::create([
      'format' => 'test_format',
      'name' => 'Test Format',
    ]);
    $filter_format->save();

    // Create editor.
    $editor = Editor::create([
      'format' => 'test_format',
      'editor' => 'ckeditor5',
      'settings' => [
        'toolbar' => [
          'items' => ['table', 'tableAdvanced'],
        ],
        'plugins' => [
          'editor_advanced_table_table_advanced' => [
            'allow_classes' => TRUE,
            'allow_id' => TRUE,
            'allow_language_direction' => FALSE,
          ],
        ],
      ],
    ]);
    $editor->save();

    $loaded_editor = Editor::load('test_format');
    $settings = $loaded_editor->getSettings();

    $this->assertArrayHasKey('editor_advanced_table_table_advanced', $settings['plugins']);
    $this->assertTrue($settings['plugins']['editor_advanced_table_table_advanced']['allow_id']);
    $this->assertTrue($settings['plugins']['editor_advanced_table_table_advanced']['allow_classes']);
    $this->assertFalse($settings['plugins']['editor_advanced_table_table_advanced']['allow_language_direction']);
  }

  /**
   * Tests plugin provides default configuration.
   */
  public function testPluginDefaultConfiguration(): void {
    $manager = $this->container->get('plugin.manager.ckeditor5.plugin');
    $plugin = $manager->createInstance('editor_advanced_table_table_advanced');
    $this->assertInstanceOf(TableAdvanced::class, $plugin);

    $config = $plugin->defaultConfiguration();
    $this->assertArrayHasKey('allow_classes', $config);
    $this->assertArrayHasKey('allow_id', $config);
    $this->assertArrayHasKey('allow_language_direction', $config);
    $this->assertTrue($config['allow_classes']);
    $this->assertTrue($config['allow_id']);
    $this->assertTrue($config['allow_language_direction']);
  }

  /**
   * Tests plugin definition has correct elements.
   */
  public function testPluginElements(): void {
    $manager = $this->container->get('plugin.manager.ckeditor5.plugin');
    $definition = $manager->getDefinition('editor_advanced_table_table_advanced');

    $elements = $definition->getElements();
    $this->assertContains('<table class id dir>', $elements);
  }

  /**
   * Tests plugin requires ckeditor5_table.
   */
  public function testPluginRequiresTablePlugin(): void {
    $manager = $this->container->get('plugin.manager.ckeditor5.plugin');
    $definition = $manager->getDefinition('editor_advanced_table_table_advanced');

    $conditions = $definition->getConditions();
    $this->assertArrayHasKey('plugins', $conditions);
    $this->assertContains('ckeditor5_table', $conditions['plugins']);
  }

  /**
   * Tests getDynamicPluginConfig returns correct structure.
   */
  public function testGetDynamicPluginConfig(): void {
    // Create filter format.
    $filter_format = FilterFormat::create([
      'format' => 'dynamic_test',
      'name' => 'Dynamic Test',
    ]);
    $filter_format->save();

    // Create editor with specific configuration.
    $editor = Editor::create([
      'format' => 'dynamic_test',
      'editor' => 'ckeditor5',
      'settings' => [
        'toolbar' => [
          'items' => ['insertTable'],
        ],
        'plugins' => [
          'editor_advanced_table_table_advanced' => [
            'allow_classes' => FALSE,
            'allow_id' => TRUE,
            'allow_language_direction' => TRUE,
          ],
        ],
      ],
    ]);
    $editor->save();

    $manager = $this->container->get('plugin.manager.ckeditor5.plugin');
    $plugin = $manager->createInstance('editor_advanced_table_table_advanced', [
      'allow_classes' => FALSE,
      'allow_id' => TRUE,
      'allow_language_direction' => TRUE,
    ]);
    $this->assertInstanceOf(TableAdvanced::class, $plugin);

    $config = $plugin->getDynamicPluginConfig([], $editor);

    $this->assertArrayHasKey('tableAdvanced', $config);
    $this->assertFalse($config['tableAdvanced']['allowClasses']);
    $this->assertTrue($config['tableAdvanced']['allowId']);
    $this->assertTrue($config['tableAdvanced']['allowLanguageDirection']);
  }

  /**
   * Tests table attributes are preserved in HTML.
   */
  public function testTableAttributesPreservedInHtml(): void {
    // Create filter format with HTML filter.
    $filter_format = FilterFormat::create([
      'format' => 'html_test',
      'name' => 'HTML Test',
      'filters' => [
        'filter_html' => [
          'status' => 1,
          'settings' => [
            'allowed_html' => '<table class id dir> <tr> <td> <th> <thead> <tbody>',
          ],
        ],
      ],
    ]);
    $filter_format->save();

    // Create editor.
    $editor = Editor::create([
      'format' => 'html_test',
      'editor' => 'ckeditor5',
      'settings' => [
        'toolbar' => [
          'items' => ['insertTable'],
        ],
        'plugins' => [
          'editor_advanced_table_table_advanced' => [
            'allow_classes' => TRUE,
            'allow_id' => TRUE,
            'allow_language_direction' => TRUE,
          ],
        ],
      ],
    ]);
    $editor->save();

    // Test HTML with table attributes.
    $html = '<table class="test-class" id="test-id" dir="ltr"><tr><td>Content</td></tr></table>';

    // Process through filter.
    $filtered = check_markup($html, 'html_test');

    // Verify attributes are preserved.
    $this->assertStringContainsString('class="test-class"', (string) $filtered);
    $this->assertStringContainsString('id="test-id"', (string) $filtered);
    $this->assertStringContainsString('dir="ltr"', (string) $filtered);
  }

  /**
   * Tests configuration affects allowed elements.
   */
  public function testConfigurationAffectsAllowedElements(): void {
    // Create filter format.
    $filter_format = FilterFormat::create([
      'format' => 'restricted_test',
      'name' => 'Restricted Test',
    ]);
    $filter_format->save();

    // Create editor with restricted configuration (classes disabled).
    $editor = Editor::create([
      'format' => 'restricted_test',
      'editor' => 'ckeditor5',
      'settings' => [
        'toolbar' => [
          'items' => ['insertTable'],
        ],
        'plugins' => [
          'editor_advanced_table_table_advanced' => [
            'allow_classes' => FALSE,
            'allow_id' => TRUE,
            'allow_language_direction' => TRUE,
          ],
        ],
      ],
    ]);
    $editor->save();

    $manager = $this->container->get('plugin.manager.ckeditor5.plugin');
    $plugin = $manager->createInstance('editor_advanced_table_table_advanced', [
      'allow_classes' => FALSE,
      'allow_id' => TRUE,
      'allow_language_direction' => TRUE,
    ]);
    $this->assertInstanceOf(TableAdvanced::class, $plugin);

    $config = $plugin->getDynamicPluginConfig([], $editor);

    // Verify classes are disabled.
    $this->assertFalse($config['tableAdvanced']['allowClasses']);
    $this->assertTrue($config['tableAdvanced']['allowId']);
  }

  /**
   * Tests multiple table attributes work together.
   */
  public function testMultipleTableAttributes(): void {
    // Create filter format.
    $filter_format = FilterFormat::create([
      'format' => 'multi_attr_test',
      'name' => 'Multi Attribute Test',
      'filters' => [
        'filter_html' => [
          'status' => 1,
          'settings' => [
            'allowed_html' => '<table class id dir> <tr> <td>',
          ],
        ],
      ],
    ]);
    $filter_format->save();

    // Test HTML with multiple attributes.
    $html = '<table class="table table-striped" id="data-table" dir="rtl"><tr><td>Data</td></tr></table>';

    $filtered = check_markup($html, 'multi_attr_test');

    // All attributes should be preserved.
    $this->assertStringContainsString('class="table table-striped"', (string) $filtered);
    $this->assertStringContainsString('id="data-table"', (string) $filtered);
    $this->assertStringContainsString('dir="rtl"', (string) $filtered);
  }

}
