<?php

namespace Drupal\editor_image_styles\Plugin\Filter;

use Drupal\Component\Utility\Html;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileUrlGenerator;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\Markup;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * @Filter(
 *   id = "editor_image_styles_filter",
 *   title = @Translation("Editor Image Styles"),
 *   description = @Translation("Applies the configured image style to inline and background images."),
 *   type = 1
 * )
 */
class ImageStyleFilter extends FilterBase implements ContainerFactoryPluginInterface {

  protected FileUrlGenerator $fileUrlGenerator;
  protected FileSystemInterface $fileSystem;
  protected CacheBackendInterface $cache;
  protected EntityTypeManagerInterface $entityTypeManager;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, FileUrlGenerator $fileUrlGenerator, FileSystemInterface $fileSystem, CacheBackendInterface $cache, EntityTypeManagerInterface $entityTypeManager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->fileUrlGenerator = $fileUrlGenerator;
    $this->fileSystem = $fileSystem;
    $this->cache = $cache;
    $this->entityTypeManager = $entityTypeManager;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('file_url_generator'),
      $container->get('file_system'),
      $container->get('cache.default'),
      $container->get('entity_type.manager')
    );
  }

  public function process($text, $langcode): FilterProcessResult {
    $config = \Drupal::config('editor_image_styles.settings');
    $style_name = $config->get('image_style');
    $base_path = $config->get('base_path') ?? '/sites/default/files/';
    $apply_bg = $config->get('apply_to_background');

    if (stripos($text, '<img') === FALSE && (!$apply_bg || stripos($text, 'background-image') === FALSE)) {
      return new FilterProcessResult($text);
    }

    $cid = 'editor_image_styles:' . md5($text);
    if ($cache = $this->cache->get($cid)) {
      return new FilterProcessResult(Markup::create($cache->data));
    }

    if (!$style_name) {
      return new FilterProcessResult($text);
    }

    $style = $this->entityTypeManager->getStorage('image_style')->load($style_name);
    if (!$style) {
      return new FilterProcessResult($text);
    }

    $dom = @Html::load($text);
    if (!$dom) {
      return new FilterProcessResult($text);
    }

    $xpath = new \DOMXPath($dom);
    $processedFiles = [];

    foreach ($xpath->query('//img') as $img) {
      $src = $img->getAttribute('src');
      if (str_contains($src, $base_path) && preg_match('/' . preg_quote($base_path, '/') . '(.*)/', $src, $m)) {
        $file_path = $m[1];
        $file_uri = 'public://' . $file_path;
        $extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));

        if (in_array($extension, ['svg', 'gif'])) {
          continue;
        }

        if (!isset($processedFiles[$file_uri])) {
          $real_path = $this->fileSystem->realpath($file_uri);
          $processedFiles[$file_uri] = $real_path && file_exists($real_path);
        }

        if ($processedFiles[$file_uri]) {
          $styled_url = $this->fileUrlGenerator->generateAbsoluteString($style->buildUrl($file_uri));
          $img->setAttribute('src', $styled_url);
        }
      }
    }

    if ($apply_bg) {
      foreach ($xpath->query('//*[@style]') as $node) {
        $style_attr = $node->getAttribute('style');
        if (str_contains($style_attr, $base_path) &&
            preg_match('/background-image\s*:\s*url\((["\']?)(' . preg_quote($base_path, '/') . '[^"\')]+)\1\)/i', $style_attr, $matches)) {

          $file_path = preg_replace('/^' . preg_quote($base_path, '/') . '/', '', urldecode($matches[2]));
          $file_uri = 'public://' . $file_path;
          $extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));

          if (in_array($extension, ['svg', 'gif'])) {
            continue;
          }

          if (!isset($processedFiles[$file_uri])) {
            $real_path = $this->fileSystem->realpath($file_uri);
            $processedFiles[$file_uri] = $real_path && file_exists($real_path);
          }

          if ($processedFiles[$file_uri]) {
            $styled_url = $this->fileUrlGenerator->generateAbsoluteString($style->buildUrl($file_uri));
            $new_style = str_replace($matches[0], 'background-image:url(' . $styled_url . ')', $style_attr);
            $node->setAttribute('style', $new_style);
          }
        }
      }
    }

    $output = Html::serialize($dom);
    $cache_tags = ['image_style:' . $style_name];

    foreach (array_keys($processedFiles) as $file_uri) {
      $file = $this->entityTypeManager->getStorage('file')->loadByProperties(['uri' => $file_uri]);
      if ($file) {
        $file = reset($file);
        $cache_tags[] = 'file:' . $file->id();
      }
    }

    $this->cache->set($cid, $output, CacheBackendInterface::CACHE_PERMANENT, $cache_tags);

    return new FilterProcessResult(Markup::create($output));
  }

}
