<?php

namespace Drupal\editoria11y\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\editoria11y\DashboardInterface;

/**
 * Provides route responses for the Editoria11y module.
 */
final class DashboardController extends ControllerBase {
  /**
   * Dashboard property.
   *
   * @var \Drupal\editoria11y\Dashboard
   */
  protected $dashboard;

  /**
   * Constructs a DashboardController object.
   *
   * @param \Drupal\editoria11y\DashboardInterface $dashboard
   *   Dashboard property.
   */
  public function __construct(DashboardInterface $dashboard) {
    $this->dashboard = $dashboard;
  }

  /**
   * {@inheritdoc}
   */
  public static function create($container) {
    return new self(
      $container->get('editoria11y.dashboard'),
    );
  }

  /**
   * Get a list of export links.
   *
   * @return array
   *   A simple renderable array.
   */
  public function getExportLinks(): array {

    /** @var \Drupal\Core\Routing\RouteProviderInterface $route_provider */
    // @phpstan-ignore-next-line
    $route_provider = \Drupal::service('router.route_provider');

    $pages = $route_provider->getRoutesByNames(['view.editoria11y_export.pages']);
    $dismissals = $route_provider->getRoutesByNames(['view.editoria11y_export.dismissals']);
    $issues = $route_provider->getRoutesByNames(['view.editoria11y_export.results']);

    $links = [];
    if (count($pages) === 1) {
      $links[] = Link::createFromRoute($this->t('Export pages with issues'), 'view.editoria11y_export.pages')->toString();
    }
    if (count($issues) === 1) {
      $links[] = Link::createFromRoute($this->t('Export alerts'), 'view.editoria11y_export.results')->toString();
    }
    if (count($dismissals) === 1) {
      $links[] = Link::createFromRoute($this->t('Export dismissals'), 'view.editoria11y_export.dismissals')->toString();
    }

    if (count($links) === 0) {
      return [];
    }
    $return = [];
    $return[] = [
      '#type' => 'html_tag',
      '#tag' => 'h2',
      '#value' => $this->t('Export results'),
    ];
    $list = [];
    foreach ($links as $link) {
      $list[] = $link;
    }
    $return[] = [
      '#theme' => 'item_list',
      '#list_type' => 'ul',
      '#items' => $list,
    ];
    return $return;

  }

  /**
   * Page: summary dashboard with three panels.
   *
   * @return array
   *   A simple renderable array.
   */
  public function dashboard(): array {
    return [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['layout-container'],
      ],
      [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['layout-container', 'layout-row'],
        ],
        [
          '#type' => 'container',
          '#attributes' => [
            'class' => ['layout-column', 'layout-column--half', 'ed11y-results-view'],
          ],
          [
            '#type' => 'html_tag',
            '#tag' => 'h2',
            '#value' => $this->t('Top issues', [], ['context' => 'problems']),
          ],
          [
            views_embed_view('editoria11y_results', 'block_top_issues'),
          ],
        ],
        [
          '#type' => 'container',
          '#attributes' => [
            'class' => ['layout-column', 'layout-column--half', 'ed11y-pages-view'],
          ],
          [
            '#type' => 'html_tag',
            '#tag' => 'h2',
            '#value' => $this->t('Pages with the most issues', [], ['context' => 'problems']),
          ],
          [
            views_embed_view('editoria11y_pages', 'block_1'),
          ],
        ],
      ],
      [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['layout-container'],
        ],

          [
            '#type' => 'html_tag',
            '#tag' => 'h2',
            '#value' => $this->t('Recent issues', [], ['context' => 'problems']),
          ],
          [
            views_embed_view('editoria11y_results', 'block_recent_issues'),
          ],

      ],
        [
          '#type' => 'container',
          '#attributes' => [
            'class' => ['layout-container', 'ed11y-dismissals-view'],
          ],
            [
              '#type' => 'html_tag',
              '#tag' => 'h2',
              '#value' => $this->t('Recent dismissals'),
            ],
            [
              views_embed_view('editoria11y_dismissals', 'recent_dismissals'),
            ],
        ],
        [
          '#type' => 'container',
          [
            '#type' => 'container',
            '#attributes' => [
              'class' => ['layout-container'],
            ],
            [
              $this->getExportLinks(),
            ],
          ],

        ],
    ];
  }

}
