<?php

namespace Drupal\editoria11y_si\Plugin\QueueWorker;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\State\State;
use Drupal\purge\Plugin\Purge\Invalidation\InvalidationsServiceInterface;
use Drupal\purge\Plugin\Purge\Processor\ProcessorsServiceInterface;
use Drupal\purge\Plugin\Purge\Purger\PurgersServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Editoria11y SI Broken Links cron event queue worker.
 *
 * @QueueWorker (
 *   id = "editoria11y_si_import_processor",
 *   title = "Editoria11y SI Import Cron Event Processor",
 *   cron = {
 *     "time" = 60,
 *     "lease_time" = 60,
 *   }
 * )
 */
class Editoria11ySiImportCronEventProcessor extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Construct function for Editoria11ySiImportCronEventProcessor.
   *
   * @param array $configuration
   *   Configuration to pass to parent class.
   * @param mixed $plugin_id
   *   Plugin id to pass to parent class.
   * @param mixed $plugin_definition
   *   Plugin definition to pass to parent class.
   * @param \Drupal\Core\Queue\QueueFactory $queueFactory
   *   Drupal queue service object.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   An entity type manager instance.
   * @param \Drupal\purge\Plugin\Purge\Processor\ProcessorsServiceInterface $purgeProcessors
   *   Service providing access to loaded processors.
   * @param \Drupal\purge\Plugin\Purge\Invalidation\InvalidationsServiceInterface $purgeInvalidationFactory
   *   A service that instantiates invalidation objects on-demand.
   * @param \Drupal\purge\Plugin\Purge\Purger\PurgersServiceInterface $purgePurgers
   *   Provides the service that distributes access to one or more purgers.
   * @param \Drupal\Core\State\State $state
   *   Drupal state instance.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    private readonly QueueFactory $queueFactory,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly ProcessorsServiceInterface $purgeProcessors,
    private readonly InvalidationsServiceInterface $purgeInvalidationFactory,
    private readonly PurgersServiceInterface $purgePurgers,
    private readonly State $state,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('queue'),
      $container->get('entity_type.manager'),
      $container->get('purge.processors'),
      $container->get('purge.invalidation.factory'),
      $container->get('purge.purgers'),
      $container->get('state'),
    );
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function processItem(
    $page,
  ) {
    $values = [
      'url' => $page['url'] ?? '',
      'siteimprove_url' => $page['siteimprove_url'] ?? '',
      'raw_data' => json_encode($page['raw_data'] ?? []),
      'siteimprove_type' => $page['siteimprove_type'] ?? '',
    ];
    $contentHash = Crypt::hashBase64(json_encode($values));
    $values['content_hash'] = $contentHash;

    // Query DB for item with url.
    $siteImproveEntityManager = $this->entityTypeManager->getStorage('editoria11y_si');
    $existingId = $siteImproveEntityManager->getQuery()
      ->condition('url', $page['url'])
      ->condition('siteimprove_type', $page['siteimprove_type'])
      ->accessCheck(FALSE)
      ->execute();

    // Make sure you found something.
    if ($existingId) {
      // $existingId is an array, make sure you only load the first item.
      /** @var \Drupal\editoria11y_si\Entity\Editoria11ySi $existingEntity */
      $existingEntity = $siteImproveEntityManager->load(reset($existingId));
      // Check if the content_hash is not the same.
      $existingContentHash = $existingEntity->get('content_hash')->getString();

      // Check if the hashes match or not.
      if ($existingContentHash !== $contentHash) {
        // Don't match, so need to update the data.
        foreach ($values as $field_name => $updated_value) {
          $existingEntity->set($field_name, $updated_value);
        }
        $existingEntity->save();
      }
    }
    else {
      // New site_improve, so save it.
      $siteImproveEntityManager->create($values)->save();
    }

    if ($this->queueFactory->get('editoria11y_si_import_processor')->numberOfItems() === 1) {
      if ($values['siteimprove_type'] === 'broken_links') {
        $this->state->set('editoria11y_si_broken_links.import.last_update', date('F j, Y, g:i a'));
      }
      if ($values['siteimprove_type'] === 'misspellings') {
        $this->state->set('editoria11y_si_misspellings.import.last_update', date('F j, Y, g:i a'));
      }
      if ($values['siteimprove_type'] === 'reading_score') {
        $this->state->set('editoria11y_si_reading_score.import.last_update', date('F j, Y, g:i a'));
      }
    }
  }

}
