<?php

declare(strict_types=1);

namespace Drupal\editoria11y_si\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\State\State;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Configure editoria11y_si settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * Editoria11y SI SettingsForm construct.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory instance.
   * @param \Drupal\Core\State\State $state
   *   Drupal state system object.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request
   *   The request object.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    private readonly State $state,
    private readonly RequestStack $request,
  ) {
    parent::__construct($configFactory);
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function create(ContainerInterface $container) {
    $form = new static(
      $container->get('config.factory'),
      $container->get('state'),
      $container->get('request_stack'),
    );
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'editoria11y_si_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['editoria11y_si.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('editoria11y_si.settings');
    $form['overview'] = [
      '#type' => 'markup',
      '#markup' => 'If you NEED to delete ALL editoria11y_si, drush entity:delete editoria11y_si</p>',
    ];
    $form['siteimprove_api'] = [
      '#title' => $this->t("SiteImprove API key"),
      '#type' => 'key_select',
      '#key_filters' => ['type' => 'editoria11y_si_key'],
      '#default_value' => $config->get('siteimprove_api'),
      '#required' => TRUE,
    ];

    $domain = $this->request->getCurrentRequest()->getSchemeAndHttpHost();
    $form['domain'] = [
      '#title' => $this->t("Domain"),
      '#type' => 'textfield',
      '#default_value' => $config->get('domain') ?? $domain,
      '#description' => $this->t('Domain of website. Will strip this from SiteImprove data since internal urls often do not have this.'),
      '#required' => TRUE,
    ];

    $form['broken_links'] = [
      '#type' => 'details',
      '#description' => $this->t('<p>See the reports page at <a href="/admin/reports/editoria11y/si-broken-links">/admin/reports/editoria11y/si-broken-links</a> to see all broken links.</p>'),
      '#title' => 'Check for broken links',
      '#open' => $config->get('editoria11y_si_broken_links_import_enabled') ?? TRUE,
    ];
    $form['broken_links']['editoria11y_si_broken_links_import_enabled'] = [
      '#type' => 'select',
      '#title' => $this->t('Enable or Disable check for broken links'),
      '#options' => [
        TRUE => $this->t('Enabled'),
        FALSE => $this->t('Disabled'),
      ],
      '#default_value' => $config->get('editoria11y_si_broken_links_import_enabled') ?? TRUE,
      '#description' => $this->t("If this is disabled, broken link checking will not occur."),
      '#required' => TRUE,
    ];
    $brokenLinksLastImported = $this->state->get('editoria11y_si_broken_links.import.last_update') ?? $this->t('Never');
    $form['broken_links']['manual_import'] = [
      '#type' => 'submit',
      '#value' => $this->t('Update SiteImprove broken link data manually'),
      '#description' => $this->t('Force the data to update from SiteImprove.'),
      '#submit' => ['::startImportOfBrokenLinks'],
      '#prefix' => $this->t(
      "<div class='description'>Last successful import: @time</div>",
        ['@time' => $brokenLinksLastImported]
      ),
    ];

    $form['misspellings'] = [
      '#type' => 'details',
      '#description' => $this->t('<p>See the reports page at <a href="/admin/reports/editoria11y/si-misspellings">/admin/reports/editoria11y/si-misspellings</a> to see all misspellings.</p>'),
      '#title' => 'Check for misspellings',
      '#open' => $config->get('editoria11y_si_misspellings_import_enabled') ?? TRUE,
    ];
    $form['misspellings']['editoria11y_si_misspellings_import_enabled'] = [
      '#type' => 'select',
      '#title' => $this->t('Enable or Disable check for misspellings'),
      '#options' => [
        TRUE => $this->t('Enabled'),
        FALSE => $this->t('Disabled'),
      ],
      '#default_value' => $config->get('editoria11y_si_misspellings_import_enabled') ?? TRUE,
      '#description' => $this->t("If this is disabled, misspelling checking will not occur."),
      '#required' => TRUE,
    ];
    $misspellingsLastImported = $this->state->get('editoria11y_si_misspellings.import.last_update') ?? $this->t('Never');
    $form['misspellings']['manual_import'] = [
      '#type' => 'submit',
      '#value' => $this->t('Update SiteImprove misspellings data manually'),
      '#description' => $this->t('Force the data to update from SiteImprove.'),
      '#submit' => ['::startImportOfMisspellings'],
      '#prefix' => $this->t(
      "<div class='description'>Last successful import: @time</div>",
        ['@time' => $misspellingsLastImported]
      ),
    ];

    $form['reading_score'] = [
      '#type' => 'details',
      '#description' => $this->t('<p>Pulls in the <a href="https://help.siteimprove.com/support/solutions/articles/80000448325#h_97063023-dd4c-4237-a499-8eddf9d1c80e" target="_blank">Flesch Kincaid Grade Level</a> from SiteImprove.</p><p>See the reports page at <a href="/admin/reports/editoria11y/si-reading-score">/admin/reports/editoria11y/si-reading-score</a> to see all reading scores.</p>'),
      '#title' => 'Check for reading scores',
      '#open' => $config->get('editoria11y_si_reading_score_import_enabled') ?? TRUE,
    ];
    $form['reading_score']['editoria11y_si_reading_score_import_enabled'] = [
      '#type' => 'select',
      '#title' => $this->t('Enable or Disable check for reading scores'),
      '#options' => [
        TRUE => $this->t('Enabled'),
        FALSE => $this->t('Disabled'),
      ],
      '#default_value' => $config->get('editoria11y_si_reading_score_import_enabled') ?? TRUE,
      '#description' => $this->t("If this is disabled, reading score checking will not occur."),
      '#required' => TRUE,
    ];
    $form['reading_score']['editoria11y_si_reading_score_error_grade_level'] = [
      '#title' => $this->t("Error grade level"),
      '#type' => 'number',
      '#default_value' => $config->get('editoria11y_si_reading_score_error_grade_level') ?? '8',
      '#description' => $this->t('If a page has a reading level below this, it will be flagged on the h1 of the page.'),
      '#required' => TRUE,
    ];
    $readingScoreLastImported = $this->state->get('editoria11y_si_reading_score.import.last_update') ?? $this->t('Never');
    $form['reading_score']['manual_import'] = [
      '#type' => 'submit',
      '#value' => $this->t('Update SiteImprove reading score data manually'),
      '#description' => $this->t('Force the data to update from SiteImprove.'),
      '#submit' => ['::startImportOfReadingScores'],
      '#prefix' => $this->t(
      "<div class='description'>Last successful import: @time</div>",
        ['@time' => $readingScoreLastImported]
      ),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('editoria11y_si.settings')
      ->set('siteimprove_api', $form_state->getValue('siteimprove_api'))
      ->set('domain', $form_state->getValue('domain'))
      ->set('editoria11y_si_broken_links_import_enabled', $form_state->getValue('editoria11y_si_broken_links_import_enabled'))
      ->set('editoria11y_si_misspellings_import_enabled', $form_state->getValue('editoria11y_si_misspellings_import_enabled'))
      ->set('editoria11y_si_reading_score_import_enabled', $form_state->getValue('editoria11y_si_reading_score_import_enabled'))
      ->set('editoria11y_si_reading_score_error_grade_level', $form_state->getValue('editoria11y_si_reading_score_error_grade_level'))
      ->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * Manual import Site Improve broken links.
   */
  public function startImportOfBrokenLinks(array &$form, FormStateInterface $form_state) {
    // Save current config to restore after the import.
    $currentConfig = $this->config('editoria11y_si.settings')->getRawData();
    // Disable other imports while this is running.
    $this->config('editoria11y_si.settings')
      ->set('editoria11y_si_broken_links_import_enabled', TRUE)
      ->set('editoria11y_si_misspellings_import_enabled', FALSE)
      ->set('editoria11y_si_reading_score_import_enabled', FALSE)
      ->save();
    editoria11y_si_queue_cronjob();
    $this->config('editoria11y_si.settings')->setData($currentConfig)->save();

    $message = $this->t('SiteImprove broken link data updated.');
    $this->messenger()->addStatus($message);
  }

  /**
   * Manual import Site Improve misspellings.
   */
  public function startImportOfMisspellings(array &$form, FormStateInterface $form_state) {
    // Save current config to restore after the import.
    $currentConfig = $this->config('editoria11y_si.settings')->getRawData();
    // Disable other imports while this is running.
    $this->config('editoria11y_si.settings')
      ->set('editoria11y_si_broken_links_import_enabled', FALSE)
      ->set('editoria11y_si_misspellings_import_enabled', TRUE)
      ->set('editoria11y_si_reading_score_import_enabled', FALSE)
      ->save();
    editoria11y_si_queue_cronjob();
    $this->config('editoria11y_si.settings')->setData($currentConfig)->save();

    $message = $this->t('SiteImprove misspellings data updated.');
    $this->messenger()->addStatus($message);
  }

  /**
   * Manual import Site Improve reading scores.
   */
  public function startImportOfReadingScores(array &$form, FormStateInterface $form_state) {
    // Save current config to restore after the import.
    $currentConfig = $this->config('editoria11y_si.settings')->getRawData();
    // Disable other imports while this is running.
    $this->config('editoria11y_si.settings')
      ->set('editoria11y_si_broken_links_import_enabled', FALSE)
      ->set('editoria11y_si_misspellings_import_enabled', FALSE)
      ->set('editoria11y_si_reading_score_import_enabled', TRUE)
      ->save();
    editoria11y_si_queue_cronjob();
    $this->config('editoria11y_si.settings')->setData($currentConfig)->save();

    $message = $this->t('SiteImprove reading score data updated.');
    $this->messenger()->addStatus($message);
  }

}
