/**
 * @file
 * Ejector seat Javascript functions.
 *
 * Poll a Drupal site via AJAX at a specified interval to determine if the user
 * currently accessing the site still has an active session and reload the page
 * if they do not. Effectively logging the user out of the site.
 */
(function (Drupal, once) {
  Drupal.behaviors.ejectorseat = {
    attach(context, settings) {
      // Ensure that this behavior is only attached once to the window.
      once('ejectorseat', 'body', context).forEach(function (element) {
        let intervalId;
        const ejectorInterval = settings.ejectorSeat.interval
          ? settings.ejectorSeat.interval * 1000
          : 60000;
        Drupal.ejectorSeat = {
          windowFocus: true,
          overdue: false,
        };

        function ejectorCheck() {
          const ignoreFocus = settings.ejectorSeat.ignoreFocus;

          if (Drupal.ejectorSeat.windowFocus || ignoreFocus) {
            // Use native Fetch API to do the AJAX request.
            fetch(settings.ejectorSeat.url, {
              method: 'GET',
              headers: {
                'X-Requested-With': 'XMLHttpRequest',
              },
            })
              .then(function (response) {
                // If the request failed, log it to console.
                if (!response.ok) {
                  throw new Error(
                    `HTTP Error: ${response.status} ${response.statusText}`,
                  );
                }
                return response.text();
              })
              .then(function (data) {
                // Only action a result of 0. Any other response is ignored.
                if (data === '0') {
                  // If the test returns 0, the user's session has ended so
                  // refresh the page for it to take effect.
                  window.location.reload(true);
                }
              })
              .catch(function (error) {
                console.error('EjectorSeat:', error);
              });
          } else {
            Drupal.ejectorSeat.overdue = true;
          }
        }

        function startTimer() {
          intervalId = setInterval(ejectorCheck, ejectorInterval);
        }

        function restartTimer() {
          clearInterval(intervalId);
          startTimer();
        }

        // Attach the 'blur' event listener.
        element.addEventListener('blur', function () {
          Drupal.ejectorSeat.windowFocus = false;
        });

        // Attach the 'focus' event listener.
        element.addEventListener('focus', function () {
          Drupal.ejectorSeat.windowFocus = true;
          if (Drupal.ejectorSeat.overdue) {
            Drupal.ejectorSeat.overdue = false;
            ejectorCheck();
            restartTimer();
          }
        });

        startTimer();
      });
    },
  };
})(Drupal, once);
