<?php

declare(strict_types=1);

namespace Drupal\Tests\ejectorseat\Functional;

use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;

/**
 * Test inclusion of javascript file as needed and admin access to settings.
 *
 * @group ejectorseat
 */
class EjectorSeatTest extends BrowserTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'ejectorseat',
  ];

  /**
   * The recommended theme to use when running test.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * A normal user without any special access.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $user;

  /**
   * An admin user who can change settings for the module.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $admin;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->user = $this->drupalCreateUser();
    $this->admin = $this->drupalCreateUser(['administer site configuration']);
  }

  /**
   * Tests for permissions and adding js only for logged in users.
   */
  public function testJavascriptInclude() {
    // Get paths to checked urls.
    $module_path = \Drupal::service('extension.list.module')
      ->getPath('ejectorseat');
    $js_path = "$module_path/ejectorseat.js";
    $settings_path = Url::fromRoute('ejectorseat.settings');
    $check_path = Url::fromRoute('user.login_status.http', [], [
      'query' => ['_format' => 'json'],
    ]);

    // Anonymous users will not be served the javascript.
    $this->drupalGet(Url::fromRoute('<front>'));
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseNotContains($js_path);

    // Anon users can also access the check url.
    $this->drupalGet($check_path);
    $this->assertSession()->statusCodeEquals(200);

    // Anon users cannot get to settings form.
    $this->drupalGet($settings_path);
    $this->assertSession()->statusCodeEquals(403);

    // Login as normal user and confirm javascript added to page.
    $this->drupalLogin($this->user);
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseContains($js_path);

    // Logged in user can access the check url.
    $this->drupalGet($check_path);
    $this->assertSession()->statusCodeEquals(200);

    // Normal user cannot get to settings form.
    $this->drupalGet($settings_path);
    $this->assertSession()->statusCodeEquals(403);

    // Login as admin and confirm settings page is present.
    $this->drupalLogin($this->admin);
    $this->drupalGet($settings_path);
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseContains('Login status check rate');
  }

}
