<?php

namespace Drupal\ejectorseat\Form;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Administration settings form for Ejector Seat module.
 */
class EjectorseatSettings extends ConfigFormBase implements ContainerInjectionInterface {

  /**
   * Date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ejectorseat_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ejectorseat.settings'];
  }

  /**
   * Create a new EjectorseatSettings object.
   */
  public function __construct(DateFormatterInterface $date_formatter) {
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static($container->get('date.formatter'));
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('ejectorseat.settings');

    $config->set(
      'ejectorseat_interval',
      $form_state->getValue('ejectorseat_interval')
    );
    $config->set(
      'ejectorseat_background',
      $form_state->getValue('ejectorseat_background')
    );
    $config->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('ejectorseat.settings');

    $time_options = [15, 30, 60, 120, 180, 300, 600, 1200, 1800, 3600];
    $form['ejectorseat_interval'] = [
      '#type' => 'select',
      '#title' => $this->t('AJAX check rate'),
      '#options' => [0 => $this->t('disable')] + array_map(
        [$this->dateFormatter, 'formatInterval'],
        array_combine($time_options, $time_options)
      ),
      '#description' => $this->t("How often should Ejector Seat check to see if a user has been logged out? Lower rates may affect site performance."),
      '#default_value' => $config->get('ejectorseat_interval') ?? 60,
    ];
    $form['ejectorseat_background'] = [
      '#type' => 'radios',
      '#title' => $this->t('Run AJAX check in background'),
      '#options' => [
        0 => $this->t('<strong>FALSE:</strong> Only run the AJAX check when the browser window or tab is active. (better performance)'),
        1 => $this->t('<strong>TRUE:</strong> Run the AJAX check even if the browser window or tab is in the background. (better security, CONSIDERABLY worse performance)'),
      ],
      '#default_value' => $config->get('ejectorseat_background') ?? 0,
    ];

    return parent::buildForm($form, $form_state);
  }

}
