<?php

namespace Drupal\elastic_email\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Messenger\Messenger;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\elastic_email\Service\ElasticEmailManager;
use ElasticEmail\ApiException;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber for the Elastic Email module.
 */
class InitSubscriber implements EventSubscriberInterface {

  /**
   * The Elastic Email API service.
   *
   * @var \Drupal\elastic_email\Service\ElasticEmailManager
   */
  protected ElasticEmailManager $elasticEmailManager;

  /**
   * Messenger.
   *
   * @var \Drupal\Core\Messenger\Messenger
   */
  protected Messenger $messenger;

  /**
   * The module configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * The current user account.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The current route.
   *
   * @var \Drupal\Core\Routing\AdminContext
   */
  protected AdminContext $route;

  /**
   * Constructs a InitSubscriber object.
   *
   * @param \Drupal\elastic_email\Service\ElasticEmailManager $elasticEmailManager
   *   The Elastic Email API service.
   * @param \Drupal\Core\Messenger\Messenger $messenger
   *   Messenger.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user account.
   * @param \Drupal\Core\Routing\AdminContext $route
   *   The current route.
   */
  public function __construct(ElasticEmailManager $elasticEmailManager, Messenger $messenger, ConfigFactoryInterface $configFactory, AccountProxyInterface $currentUser, AdminContext $route) {
    $this->elasticEmailManager = $elasticEmailManager;
    $this->messenger = $messenger;
    $this->config = $configFactory->get('elastic_email.settings');
    $this->currentUser = $currentUser;
    $this->route = $route;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [KernelEvents::REQUEST => ['onEvent', 0]];
  }

  /**
   * Callback for kernel events.
   */
  public function onEvent(): void {
    if (!$this->currentUser->hasPermission('administer site configuration')
      || !$this->route->isAdminRoute()
      || !($creditLowThreshold = $this->config->get('credit_low_threshold'))) {
      return;
    }
    try {
      $rawData = $this->elasticEmailManager->getAccountInfo();
      if ($rawData->getEmailCredits() >= $creditLowThreshold) {
        $this->messenger->addMessage(new TranslatableMarkup('Your Elastic Email credit is getting low - currently at %credit %currency', [
          '%credit' => $rawData->getEmailCredits(),
          '%currency' => '?',
        ]), 'warning');
      }
    }
    catch (ApiException) {
      // Deliberately ignored exception.
    }
  }

}
