<?php

namespace Drupal\elastic_email\Plugin\Mail;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Mail\MailInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueInterface;
use Drupal\elastic_email\Service\ElasticEmailManager;
use ElasticEmail\ApiException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Modify the drupal mail system to use Elastic Email to send emails.
 *
 * @Mail(
 *   id = "elastic_email_mail",
 *   label = @Translation("Elastic Email Mailer"),
 *   label_singular = @Translation("Elastic Email Mailer"),
 *   label_plural = @Translation("Elastic Email Mailers"),
 *   label_count = @PluralTranslation(
 *     singular = @Translation("elastic email mailer"),
 *     plural = @Translation("elastic email mailers")
 *   ),
 *   description = @Translation("Sends emails via Elastic Email.")
 * )
 */
class ElasticEmailMailSystem extends PluginBase implements MailInterface, ContainerFactoryPluginInterface {

  /**
   * This module's config.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $moduleConfig;

  /**
   * This site's config.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $siteConfig;

  /**
   * Mail logger instance.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $loggerMail;

  /**
   * Queue logger instance.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $loggerQueue;

  /**
   * Queue instance.
   *
   * @var \Drupal\Core\Queue\QueueInterface
   */
  protected QueueInterface $queue;

  /**
   * Elastic Email manager instance.
   *
   * @var \Drupal\elastic_email\Service\ElasticEmailManager
   */
  protected ElasticEmailManager $elasticEmailManager;

  /**
   * Constructs a ElasticEmailMailSystem object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory instance.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerChannelFactory
   *   Logger factory instance.
   * @param \Drupal\Core\Queue\QueueFactory $queueFactory
   *   Queue factory instance.
   * @param \Drupal\elastic_email\Service\ElasticEmailManager $elasticEmailManager
   *   Elastic Email manager instance.
   */
  final public function __construct(array $configuration, $plugin_id, $plugin_definition, ConfigFactoryInterface $configFactory, LoggerChannelFactoryInterface $loggerChannelFactory, QueueFactory $queueFactory, ElasticEmailManager $elasticEmailManager) {
    $this->moduleConfig = $configFactory->get('elastic_email.settings');
    $this->siteConfig = $configFactory->get('elastic_email.settings');
    $this->loggerMail = $loggerChannelFactory->get('elastic_email');
    $this->loggerQueue = $loggerChannelFactory->get('elastic_email_queue');
    $this->queue = $queueFactory->get('elastic_email_process_queue');
    $this->elasticEmailManager = $elasticEmailManager;
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('logger.factory'),
      $container->get('queue'),
      $container->get('elastic_email.api')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function format(array $message): array {
    // Join the body array into one string.
    $message['body'] = implode("\n\n", $message['body']);

    return $message;
  }

  /**
   * {@inheritdoc}
   */
  public function mail(array $message): bool {
    // If queueing is available and enabled, queue the message.
    if ($this->moduleConfig->get('queue_enabled')) {
      $this->queue->createItem($message);
      $this->queue->createQueue();

      $this->loggerQueue->info('Message added to the Queue - no. of messages: ' . $this->queue->numberOfItems());

      return TRUE;
    }
    // Otherwise send the message directly.
    return $this->send($message);
  }

  /**
   * Performs the actual sending of the message to the Elastic Email service.
   *
   * This code was originally based on the 'sendElasticEmail' code snippet from
   * the Elastic Email website.
   *
   * The $elastic_username and $api_key variables are generally not required, as
   * they they are specified in the module configuration. However, you may
   * supply these parameters if you wish to override the configuration values.
   *
   * You must provide either the $subject or $body_text parameter. That is, it
   * is not possible to send an empty email.
   *
   * If this method succeeds, an array will be returned with the following
   * elements:
   * - array['success']['tx_id']: the transaction id returned by Elastic Email
   * - array['success']['to']: semi-colon separated list of recipients
   * - array['success']['msg']: user-friendly message
   *
   * If there's an error, an array will be returned with the following element:
   * - array['error'] : The error message returned by Elastic Email, or an
   * error
   * message from this module if a pre-condition was not met (e.g. missing
   * required parameters).
   *
   * @param string $from
   *   The 'from' (sender) email address.
   * @param string $to
   *   The semi-colon separated list of recipient email addresses.
   * @param string $subject
   *   The subject line.
   * @param string|null $body_text
   *   The plain-text body of the message.
   * @param string|null $body_html
   *   The html-text body of the message.
   *
   * @return array
   *   Returns an array with either a 'success' or 'error' elements. See main
   *   function description for details. Note that the error message text will
   *   have already been passed through $this->t().
   *
   * @todo Provide support for HTML-based email and attachments?
   */
  public function elasticEmailSend(string $from, string $to, string $subject = '', ?string $body_text = NULL, ?string $body_html = NULL): array {
    $username = $this->moduleConfig->get('username');
    $api_key = $this->moduleConfig->get('api_key');

    $result = [];
    if (empty($username) || empty($api_key)) {
      $result['error'] = $this->t('Unable to send email to Elastic Email because username or API key not specified.');
    }
    elseif (empty($from) || empty($to) || (empty($subject) && empty($body_text))) {
      $result['error'] = $this->t('Unable to send email because some required email parameters are empty.');
    }

    if (isset($result['error'])) {
      return $result;
    }

    // Set the reply to if enabled.
    $use_reply_to = $this->moduleConfig->get('use_reply_to');
    $reply_to = ($use_reply_to) ? $this->moduleConfig->get('reply_to_email') : NULL;
    $defaultChannel = $this->moduleConfig->get('use_default_channel') ?
      $this->moduleConfig->get('default_channel') :
      NULL;
    $recipients = explode(';', $to);
    $response = NULL;
    try {
      $response = $this->elasticEmailManager->sendEmail($subject, $from, $reply_to, $recipients, $defaultChannel, $body_html, $body_text);
    }
    catch (ApiException $e) {
      $this->messenger()->addError($e->getMessage());
    }

    if ($response === NULL) {
      $result['error'] = $this->t('Error: no response (or empty response) received from Elastic Email service.');
    }
    elseif (!preg_match('/[0-9a-z]{8}-[0-9a-z]{4}-[0-9a-z]{4}-[0-9a-z]{4}-[0-9a-z]{12}/', $response->getTransactionId())) {
      $result['error'] = 'error message';
    }
    else {
      // Message was successfully delivered.
      $result['success']['msg'] = $this->t('Success [@tx_id]; message sent to: @recipients',
        [
          '@tx_id' => $response->getTransactionId(),
          '@recipients' => $recipients,
        ]);

      $result['success']['tx_id'] = $response->getTransactionId();
      $result['success']['recipients'] = $recipients;
    }

    return $result;
  }

  /**
   * Email sending function, called from job queue, or directly.
   *
   * @param array $message
   *   Standard Drupal email message object.
   *
   * @return bool
   *   TRUE if message delivered; FALSE otherwise.
   */
  public function send(array $message = []): bool {
    // If there's no 'from', then use the default site email.
    if (empty($message['from'])) {
      $from = $this->siteConfig->get('mail');
      if (!empty($from)) {
        $message['from'] = $from;
      }
    }

    // Parse $message['from'] into $from and $from_name if in full RFC format.
    if (preg_match('~^"?([^"]+)"? <\s*(.+)\s*>$~', $message['from'], $matches)) {
      [, $from] = $matches;
    }
    else {
      $from = $message['from'];
    }

    // Array to hold the set of email recipients.
    $recipients = [];

    // Parse the various fields that can contain email recipients.
    $this->parseRecipient($recipients, $message['to']);
    if (isset($message['headers']['Cc'])) {
      $this->parseRecipient($recipients, $message['headers']['Cc']);
    }
    if (isset($message['headers']['Bcc'])) {
      $this->parseRecipient($recipients, $message['headers']['Bcc']);
    }

    // Concatenate recipients to a semi-colon separated string.
    $to = '';
    if (count($recipients)) {
      $to = implode('; ', $recipients);
    }

    // Check the header content type to see if email is plain text, if not we
    // send as HTML.
    $is_html = !str_contains($message['headers']['Content-Type'], 'text/plain');

    // Attempt to send the message.
    $body_text = ($is_html ? NULL : $message['body']);
    $body_html = ($is_html ? $message['body'] : NULL);
    $result = $this->elasticEmailSend($from, $to, $message['subject'], $body_text, $body_html);

    if (isset($result['error'])) {
      // If there's an error, log it.
      $this->loggerMail->critical('Failed to send email.  Reason: ' . $result['error']);
    }

    // If success, only log if the ELASTIC_EMAIL_LOG_SUCCESS variable is TRUE.
    if (isset($result['success']) && $this->siteConfig->get('log_success')) {
      $this->loggerMail->info('Email sent successfully: ' . $result['success']['msg']);
    }

    return (isset($result['success']) && $result['success']);
  }

  /**
   * Parse comma separated list of email addresses and add them to array.
   *
   * Given a comma-delimited list of email addresses in the $to parameter, parse
   * the addresses and add to the $recipients array.
   *
   * @param array $recipients
   *   A passed-by-reference array holding recipient email addresses.
   * @param string $to
   *   A comma-delimited list of email addresses.
   */
  protected function parseRecipient(array &$recipients, string $to): void {
    if (!$to) {
      return;
    }

    // Trim any whitespace.
    $to = trim($to);
    if (!empty($to)) {
      // Explode on comma.
      $parts = explode(',', $to);
      foreach ($parts as $part) {
        // Again, trim any whitespace.
        $part = trim($part);
        if (!empty($part)) {
          $recipients[] = $part;
        }
      }
    }
  }

}
