<?php

namespace Drupal\elastic_email\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use ElasticEmail\Api\EmailsApi;
use ElasticEmail\Api\StatisticsApi;
use ElasticEmail\Api\SubAccountsApi;
use ElasticEmail\Configuration;
use ElasticEmail\Model\BodyPart;
use ElasticEmail\Model\EmailContent;
use ElasticEmail\Model\EmailData;
use ElasticEmail\Model\EmailMessageData;
use ElasticEmail\Model\EmailRecipient;
use ElasticEmail\Model\EmailSend;
use ElasticEmail\Model\Options;
use ElasticEmail\Model\SubAccountInfo;
use GuzzleHttp\ClientInterface;

/**
 * Elastic Email service to interact with the API.
 */
class ElasticEmailManager {

  /**
   * The module config.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * The Guzzle HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The Elastic Email API.
   *
   * @var \ElasticEmail\Api\EmailsApi|null
   */
  protected ?EmailsApi $emailApi = NULL;

  /**
   * The Elastic Statistics API.
   *
   * @var \ElasticEmail\Api\StatisticsApi|null
   */
  protected ?StatisticsApi $statisticApi = NULL;

  /**
   * The Elastic SubAccounts API.
   *
   * @var \ElasticEmail\Api\SubAccountsApi|null
   */
  protected ?SubAccountsApi $subaccountApi = NULL;

  /**
   * Constructs the ElasticEmailManager service.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The Guzzle HTTP client.
   */
  public function __construct(ConfigFactoryInterface $configFactory, ClientInterface $httpClient) {
    $this->config = $configFactory->get('elastic_email.settings');
    $this->httpClient = $httpClient;
  }

  /**
   * Initializes and returns the Elastic Email Api.
   *
   * @return \ElasticEmail\Api\EmailsApi
   *   The Elastic Email Api.
   */
  protected function emailApi(): EmailsApi {
    if ($this->emailApi === NULL) {
      $config = Configuration::getDefaultConfiguration()
        ->setApiKey('X-ElasticEmail-ApiKey', $this->config->get('api_key'));
      $this->emailApi = new EmailsApi($this->httpClient, $config);
    }
    return $this->emailApi;
  }

  /**
   * Initializes and returns the Elastic Statistic Api.
   *
   * @return \ElasticEmail\Api\StatisticsApi
   *   The Elastic Statistic Api.
   */
  protected function statisticApi(): StatisticsApi {
    if ($this->statisticApi === NULL) {
      $config = Configuration::getDefaultConfiguration()
        ->setApiKey('X-ElasticEmail-ApiKey', $this->config->get('api_key'));
      $this->statisticApi = new StatisticsApi($this->httpClient, $config);
    }
    return $this->statisticApi;
  }

  /**
   * Initializes and returns the Elastic SubAccount Api.
   *
   * @return \ElasticEmail\Api\SubAccountsApi
   *   The Elastic SubAccount Api.
   */
  protected function subaccountApi(): SubAccountsApi {
    if ($this->subaccountApi === NULL) {
      $config = Configuration::getDefaultConfiguration()
        ->setApiKey('X-ElasticEmail-ApiKey', $this->config->get('api_key'));
      $this->subaccountApi = new SubaccountsApi($this->httpClient, $config);
    }
    return $this->subaccountApi;
  }

  /**
   * Sends an email to recipients using the API.
   *
   * @param string $subject
   *   The subject.
   * @param string $from
   *   The from address.
   * @param string|null $reply_to
   *   The reply to address, or NULL if not to be used.
   * @param array $recipients
   *   The list of recipient addresses.
   * @param string|null $channel
   *   The channel name, or NULL if not to be used.
   * @param string|null $body_html
   *   The body text in HTML, or NULL if not to be used.
   * @param string|null $body_plain
   *   The plain body text, or NULL if not to be used.
   *
   * @return \ElasticEmail\Model\EmailSend
   *   The email send response.
   *
   * @throws \ElasticEmail\ApiException
   */
  public function sendEmail(string $subject, string $from, ?string $reply_to, array $recipients, ?string $channel, ?string $body_html, ?string $body_plain): EmailSend {
    $recipientList = [];
    foreach ($recipients as $recipient) {
      $recipientList[] = new EmailRecipient(['email' => $recipient]);
    }

    $body = [];
    if ($body_html !== NULL) {
      $body[] = new BodyPart([
        'content_type' => 'HTML',
        'content' => $body_html,
      ]);
    }
    if ($body_plain !== NULL) {
      $body[] = new BodyPart([
        'content_type' => 'PlainText',
        'content' => $body_plain,
      ]);
    }

    $content = [
      'body' => $body,
      'from' => $from,
      'subject' => $subject,
    ];
    if ($reply_to !== NULL) {
      $content['reply_to'] = $reply_to;
    }

    $options = [];
    if ($channel !== NULL) {
      $options['channel_name'] = $channel;
    }

    $email_message_data = new EmailMessageData([
      "recipients" => $recipientList,
      "content" => new EmailContent($content),
      "options" => new Options($options),
    ]);
    return $this->emailApi()->emailsPost($email_message_data);
  }

  /**
   * Gets an email data object.
   *
   * @param string $msgId
   *   The email message ID to receive.
   *
   * @return \ElasticEmail\Model\EmailData
   *   The email data object.
   *
   * @throws \ElasticEmail\ApiException
   */
  public function getEmailView(string $msgId): EmailData {
    return $this->emailApi()->emailsByMsgidViewGet($msgId);
  }

  /**
   * Gets the first sub-accounts info object.
   *
   * @return \ElasticEmail\Model\SubAccountInfo
   *   The sub-accounts info object.
   *
   * @throws \ElasticEmail\ApiException
   */
  public function getAccountInfo(): SubAccountInfo {
    return $this->subaccountApi()->subaccountsGet(1)[0];
  }

  /**
   * Gets a list of channels.
   *
   * @return \ElasticEmail\Model\ChannelLogStatusSummary[]
   *   The list of channels.
   *
   * @throws \ElasticEmail\ApiException
   */
  public function getChannels(): array {
    return $this->statisticApi()->statisticsChannelsGet();
  }

}
