<?php

namespace Drupal\elastic_email\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Render\Renderer;
use Drupal\elastic_email\Service\ElasticEmailManager;
use ElasticEmail\ApiException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Default controller for the elastic_email module.
 */
class ElasticEmailController extends ControllerBase {

  /**
   * The Elastic Email API service.
   *
   * @var \Drupal\elastic_email\Service\ElasticEmailManager
   */
  protected ElasticEmailManager $elasticEmailManager;

  /**
   * Messenger.
   *
   * @var \Drupal\Core\Messenger\Messenger
   */
  protected $messenger;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\Renderer
   */
  protected Renderer $renderer;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): ElasticEmailController {
    $instance = parent::create($container);
    $instance->elasticEmailManager = $container->get('elastic_email.api');
    $instance->messenger = $container->get('messenger');
    $instance->renderer = $container->get('renderer');
    $instance->logger = $container->get('logger.factory')->get('elastic_email');
    return $instance;
  }

  /**
   * Builds the dashboards render array.
   *
   * @return array
   *   The dashboards render array.
   */
  public function dashboard(): array {
    try {
      $rawData = $this->elasticEmailManager->getAccountInfo();
      $data = [
        'credit' => $rawData->getEmailCredits(),
        'currency' => '?',
        'totalemailssent' => $rawData->getTotalEmailsSent(),
        'priceperemail' => '?',
        'reputation' => $rawData->getReputation(),
      ];
    }
    catch (ApiException) {
      $this->messenger->addError($this->t('You need to configure your Elastic Email before you can continue to use this module.'));
      $data = [];
    }
    return [
      '#theme' => 'elastic_email_dashboard',
      '#data' => $data,
      '#attached' => [
        'library' => ['elastic_email/admin'],
      ],
    ];
  }

  /**
   * Builds the render array for a given sent email.
   *
   * @param string $msgId
   *   The ID of the email to render.
   *
   * @return array
   *   The render array for the email.
   */
  public function viewEmail(string $msgId): array {
    if (empty($msgId)) {
      return [
        '#markup' => 'This is a problem with loading this email, please try again.',
      ];
    }
    $data = $this->getEmailData($msgId);
    return [
      '#theme' => 'elastic_email_view_email',
      '#title' => $this->t('View Email: "%subject"', ['%subject' => $data['subject']]),
      '#data' => $data,
      '#attached' => [
        'library' => [
          'elastic_email/admin',
        ],
      ],
    ];
  }

  /**
   * Builds the render array for a given sent email.
   *
   * @param string $msgId
   *   The ID of the email to render.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The render array for the email.
   */
  public function viewEmailContent(string $msgId): Response {
    if (empty($msgId)) {
      $build = [
        '#markup' => 'This is a problem with loading this email, please try again.',
      ];
    }
    else {
      $build = [
        'page' => [
          '#theme' => 'elastic_email_view_email_content',
          '#data' => $this->getEmailData($msgId),
        ],
      ];
    }
    $html = $this->renderer->renderRoot($build);
    $response = new Response();
    $response->setContent($html);
    return $response;
  }

  /**
   * Gets the email data for a specific message id.
   *
   * @param string $msgId
   *   The email message id.
   *
   * @return array
   *   The email data.
   */
  protected function getEmailData(string $msgId): array {
    $data = &drupal_static(__METHOD__);
    $cid = 'elastic_email:' . __FUNCTION__ . ':' . $msgId;

    if ($cache = $this->cache()->get($cid)) {
      $data = $cache->data;
    }
    else {
      try {
        $rawData = $this->elasticEmailManager->getEmailView($msgId);
        if (($preview = $rawData->getPreview()) && $status = $rawData->getStatus()) {
          $data = [
            'msgId' => $msgId,
            'date' => $status->getDate(),
            'statusname' => $status->getStatusName(),
            'to' => $status->getTo(),
            'from' => $preview->getFrom(),
            'subject' => $preview->getSubject(),
            'body' => $preview->getBody(),
          ];
          $this->cache()->set($cid, $data, 60 * 60 * 6);
        }
        else {
          $message = 'There has been a problem with decoding the received email message.';
          $this->logger->error($message);
          $data['body'] = $message;
        }
      }
      catch (ApiException $e) {
        $message = $this->t('There was a problem with retrieving email message with id @msgId. The error message was: @errMsg', [
          '@msgId' => $msgId,
          '@errMsg' => $e->getMessage(),
        ]);
        $this->logger->error($message);
        $data['body'] = 'There has been a problem with retrieving your email message.';
      }
    }
    return $data;
  }

}
