<?php

namespace Drupal\elastic_email\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\Messenger;
use Drupal\elastic_email\Plugin\Mail\ElasticEmailMailSystem;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form to test sending an email with the Elastic Email module.
 */
class ElasticEmailSendTest extends FormBase {

  /**
   * Messenger.
   *
   * @var \Drupal\Core\Messenger\Messenger
   */
  protected $messenger;

  /**
   * Mail sender instance.
   *
   * @var \Drupal\elastic_email\Plugin\Mail\ElasticEmailMailSystem
   */
  protected ElasticEmailMailSystem $sender;

  /**
   * Config factory instance.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $config;

  /**
   * Date formatter instance.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * System time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * Constructs a ElasticEmailSendTest object.
   *
   * @param \Drupal\Core\Messenger\Messenger $messenger
   *   Messenger.
   * @param \Drupal\elastic_email\Plugin\Mail\ElasticEmailMailSystem $sender
   *   Mail sender instance.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config
   *   Config factory instance.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   Date formatter instance.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   System time service.
   */
  public function __construct(Messenger $messenger, ElasticEmailMailSystem $sender, ConfigFactoryInterface $config, DateFormatterInterface $date_formatter, TimeInterface $time) {
    $this->messenger = $messenger;
    $this->sender = $sender;
    $this->config = $config;
    $this->dateFormatter = $date_formatter;
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): ElasticEmailSendTest {
    return new ElasticEmailSendTest(
      $container->get('messenger'),
      $container->get('plugin.manager.mail')
        ->createInstance('elastic_email_mail'),
      $container->get('config.factory'),
      $container->get('date.formatter'),
      $container->get('datetime.time')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'elastic_email_send_test';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['elastic_email_test_email_to'] = [
      '#type' => 'textfield',
      '#size' => 40,
      '#title' => $this->t('Email address to send a test email to'),
      '#description' => $this->t('Enter the email address that you would like to send a test email to.'),
      '#required' => TRUE,
      '#default_value' => $this->config->get('system.site')->get('mail'),
    ];

    $form['elastic_email_test_email_subject'] = [
      '#type' => 'textfield',
      '#size' => 100,
      '#title' => $this->t('Test Email Subject'),
      '#description' => $this->t('Enter the subject that you would like to send with the test email.'),
      '#required' => TRUE,
      '#default_value' => $this->t('Elastic Email module: configuration test email'),
    ];

    $text_body = $this->t("This is a test of the Drupal Elastic Email module configuration.\n\nMessage generated: @time", [
      '@time' => $this->dateFormatter->format($this->time->getRequestTime(), 'custom', 'r'),
    ]);

    $form['elastic_email_test_email_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Test email body contents'),
      '#description' => $this->t('Enter the email body that you would like to send.'),
      '#default_value' => $text_body,
    ];

    $form['elastic_email_test_email_html'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Send as HTML?'),
      '#description' => $this->t('Check this to send a test email as HTML.'),
      '#default_value' => FALSE,
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => 'Submit',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $site_mail = $this->config->get('system.site')->get('mail');

    $to = $form_state->getValue(['elastic_email_test_email_to']);
    $subject = $form_state->getValue(['elastic_email_test_email_subject']);

    if ($form_state->getValue(['elastic_email_test_email_html'])) {
      $text_body = NULL;
      $html_body = $form_state->getValue(['elastic_email_test_email_body']);
    }
    else {
      $text_body = $form_state->getValue(['elastic_email_test_email_body']);
      $html_body = NULL;
    }

    $result = $this->sender->elasticEmailSend($site_mail, $to, $subject, $text_body, $html_body);

    if (isset($result['error'])) {
      // There was an error. Return error HTML.
      $this->messenger->addError($this->t('Failed to send a test email to %test_to. Got the following error: %error_msg', [
        '%test_to' => $to,
        '%error_msg' => $result['error'],
      ]));
    }
    else {
      // Success!
      $this->messenger->addMessage(
        $this->t('Successfully sent a test email to %test_to', [
          '%test_to' => $to,
        ])
      );
    }
  }

}
