<?php

/**
 * @file
 * Views hook implementations for the Search API module.
 */

use Drupal\Component\Render\FormattableMarkup;

/**
 * Implements hook_views_data().
 */
function elasticsearch_connector_views_views_data() {
  $data = [];
  $clientManager = \Drupal::service('elasticsearch_connector.client_manager');
  /** @var \Drupal\elasticsearch_connector\ClusterManager $cluster_manager */
  $cluster_manager = \Drupal::service('elasticsearch_connector.cluster_manager');

  foreach ($cluster_manager->loadAllClusters(FALSE) as $cluster) {
    $elasticsearchClient = $clientManager->getClientForCluster($cluster);
    if ($elasticsearchClient->isClusterOk()) {
      $indices = $elasticsearchClient->indices()->stats();
      // @todo Handle aliases also, not only indices.
      if (!empty($indices['indices'])) {
        foreach ($indices['indices'] as $index_name => $index_info) {
          // Ignore system indices.
          if (strpos($index_name, '.') === 0) {
            continue;
          }

          $name = new FormattableMarkup(
            '@cluster (@index_name)', [
              '@cluster' => $cluster->name,
              '@index_name' => $index_name,
            ]
          );
          $base_table = 'elsv__' . $cluster->cluster_id . '__' . $index_name;

          $data[$base_table]['table']['group'] = t('Elasticsearch');
          $data[$base_table]['table']['base'] = [
            'index' => $index_name,
            'cluster_id' => $cluster->cluster_id,
            'title' => t('Cluster :name', [':name' => $name]),
            'help' => t('Searches the site with the Elasticsearch search engine for !name', ['!name' => $name]),
            'query_id' => 'elasticsearch_connector_views_query',
          ];

          $indexMapping = $elasticsearchClient->indices()->getMapping(['index' => $index_name]);
          // Get the list of the fields in index directly from Elasticsearch.
          if (!empty($indexMapping[$index_name]['mappings']['properties'])) {
            _elasticsearch_connector_views_handle_fields($base_table, $data, $indexMapping[$index_name]['mappings']['properties']);
          }

          // Keyword field.
          $data[$base_table]['keyword'] = [
            'title' => t('Search'),
            'help' => t('Fulltext search'),
            'filter' => [
              'id' => 'elasticsearch_connector_views_fulltext_filter',
            ],
          ];

          // Snippet field.
          $data[$base_table]['snippet'] = [
            'title' => t('Snippet'),
            'help' => t('Search snippet'),
            'field' => [
              'handler' => 'elasticsearch_connector_views_snippet_handler_field',
              'click sortable' => TRUE,
            ],
          ];

          // Score field.
          $data[$base_table]['score'] = [
            'title' => t('Score'),
            'help' => t('Score'),
            'field' => [
              'id' => 'elasticsearch_connector_views_standard',
              'click sortable' => TRUE,
            ],
          ];
        }
      }
    }
  }

  return $data;
}

/**
 * Handle the fields mapping and handle nested data types.
 *
 * @param string $base_table
 *   The base table value.
 * @param array $data
 *   Data array.
 * @param array $fields
 *   Fields array.
 * @param string $base_field_name
 *   Base field name.
 */
function _elasticsearch_connector_views_handle_fields($base_table, &$data, $fields, $base_field_name = '') {
  if (!empty($fields)) {
    foreach ($fields as $field_name => $field) {
      // @todo Restrict some fields if needed.
      // @todo Handle boolean.
      // @todo Handle the cases with analyzed and not analyzed.
      if (empty($field['type']) && isset($field['properties'])) {
        $field_type = 'object';
      }
      else {
        $field_type = $field['type'];
      }

      $filter_handler = 'elasticsearch_connector_views_standard';
      $field_handler = 'elasticsearch_connector_views_standard';
      $set = TRUE;
      switch ($field_type) {
        case 'object':
          if (!empty($field['properties'])) {
            _elasticsearch_connector_views_handle_fields($base_table, $data, $field['properties'], $base_field_name . $field_name . '.');
          }
          $set = FALSE;
          break;

        case 'date':
          $filter_handler = 'elasticsearch_connector_views_date';
          $field_handler = 'elasticsearch_connector_views_date';
          break;

        case 'boolean':
          $filter_handler = 'elasticsearch_connector_views_boolean';
          $field_handler = 'elasticsearch_connector_views_boolean';
          break;

        // @todo Handle the keyword also for the text fields!
        case 'text':
        case 'string':
        case 'keyword':
          // @todo Handle the analyser and non_analyzed fields.
          // @todo For analysed fields we need to do fulltext search.
          if (\Drupal::moduleHandler()
            ->moduleExists('views_autocomplete_filters')
          ) {
            // @todo Handle autocomplete.
            // $filter_handler = 'elasticsearch_connector_views_handler_filter_string_autocomplete';
          }
          else {
            $field_handler = 'elasticsearch_connector_views_markup';
            $filter_handler = 'elasticsearch_connector_views_string';
          }
          break;

        // Handle numeric filter type.
        case 'integer':
        case 'long':
        case 'float':
        case 'double':
          $filter_handler = 'elasticsearch_connector_views_numeric';
          $field_handler = 'elasticsearch_connector_views_numeric';
          break;
      }

      if ($set) {
        $data[$base_table][$base_field_name . $field_name] = [
          'title' => $base_field_name . $field_name,
          'help' => $base_field_name . $field_name,
          'field' => [
            'id' => $field_handler,
            'click sortable' => TRUE,
          ],
          'filter' => [
            'id' => $filter_handler,
          ],
          'sort' => [
            'id' => 'standard',
          ],
          // @todo Handle the argument class.
          'argument' => [
            'id' => 'standard',
          ],
        ];
      }
    }
  }
}
