<?php

namespace Drupal\elasticsearch_connector\ElasticSearch\Parameters\Factory;

use Drupal\search_api\Item\FieldInterface;
use Drupal\elasticsearch_connector\Event\PrepareMappingEvent;

/**
 * Helper class to build Elastic mapping configuration from a Search API Field.
 */
class MappingFactory {

  /**
   * Helper function. Get the elasticsearch mapping for a field.
   *
   * @param \Drupal\search_api\Item\FieldInterface $field
   *   The Search API Index Field that we should get a mapping for.
   *
   * @return array|null
   *   Array of settings when a known field type is provided. Null otherwise.
   */
  public static function mappingFromField(FieldInterface $field) {
    $type = $field->getType();
    $mappingConfig = NULL;

    switch ($type) {
      case 'text':
        $mappingConfig = [
          'type' => 'text',
          'fields' => [
            "keyword" => [
              "type" => 'keyword',
              'ignore_above' => 256,
            ],
          ],
        ];
        break;

      case 'uri':
      case 'string':
      case 'token':
        $mappingConfig = [
          'type' => 'keyword',
        ];
        break;

      case 'integer':
      case 'duration':
        $mappingConfig = [
          'type' => 'integer',
        ];
        break;

      case 'boolean':
        $mappingConfig = [
          'type' => 'boolean',
        ];
        break;

      case 'decimal':
        $mappingConfig = [
          'type' => 'float',
        ];
        break;

      case 'date':
        $mappingConfig = [
          'type' => 'date',
          'format' => 'strict_date_optional_time||epoch_second',
        ];
        break;

      case 'full_date':
        $mappingConfig = [
          'type' => 'date',
          'format' => 'strict_date_optional_time',
        ];
        break;

      case 'attachment':
        $mappingConfig = [
          'type' => 'attachment',
        ];
        break;

      case 'object':
        $mappingConfig = [
          'type' => 'nested',
        ];
        break;

      case 'location':
        $mappingConfig = [
          'type' => 'geo_point',
        ];
        break;
    }

    // Allow other modules to alter mapping config before we create it.
    $dispatcher = \Drupal::service('event_dispatcher');
    $prepareMappingEvent = new PrepareMappingEvent($mappingConfig, $type, $field);
    $event = $dispatcher->dispatch($prepareMappingEvent, PrepareMappingEvent::PREPARE_MAPPING);
    $mappingConfig = $event->getMappingConfig();

    return $mappingConfig;
  }

}
