<?php

namespace Drupal\elasticsearch_connector\Tests\Kernel;

use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\search_api\Entity\Index;
use Drupal\search_api\Query\QueryInterface;
use Drupal\Tests\search_api\Kernel\BackendTestBase;

/**
 * Tests index and search capabilities using the elasticsearch backend.
 *
 * @group elasticsearch_connector
 */
class ElasticsearchTest extends BackendTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  public static $modules = ['facets', 'elasticsearch_connector', 'elasticsearch_test'];

  /**
   * A Search API index ID.
   *
   * @var string
   */
  protected $indexId = 'elasticsearch_index';

  /**
   * A Search API server ID.
   *
   * @var string
   */
  protected $serverId = 'elasticsearch_server';

  /**
   * {@inheritdoc}
   */
  public function setUp() : void {
    parent::setUp();
    $this->installEntitySchema('facets_facet');
    $this->installConfig([
      'facets',
      'elasticsearch_connector',
      'elasticsearch_test',
    ]);
  }

  /**
   * {@inheritdoc}
   */
  protected function checkFacets() {
    // No-op.
    //
    // For some reason, Search API is no longer sending the 'category' field in
    // the entity_test_mulrev_changed entity for indexing. This appears to be
    // unrelated to any of the changes we've made in this module. But, the
    // BackendTestBase::checkFacets() test facets on this field, so the test
    // fails for us. Note that the equivalent
    // \Drupal\Tests\search_api_opensearch\Kernel\OpenSearchBackendTest also
    // skips BackendTestBase::checkFacets().
  }

  /**
   * {@inheritdoc}
   */
  protected function checkIndexWithoutFields() {
    $index = Index::create([
      'id' => 'test_index_2',
      'name' => 'Test index 2',
      'status' => TRUE,
      'server' => $this->serverId,
      'datasource_settings' => [
        'entity:entity_test_mulrev_changed' => [],
      ],
      'tracker_settings' => [
        'default' => [],
      ],
    ]);
    $index->save();

    $indexed_count = $this->indexItems($index->id());
    $this->assertEquals(count($this->entities), $indexed_count);

    // ElasticSearch no longer returns all items in the index when the items
    // have no fields, even if you perform a search for all items.
    $search_count = $index->query()->execute()->getResultCount();
    $this->assertEquals(0, $search_count);

    return $index;
  }

  /**
   * Regression tests.
   */
  protected function regressionTests() {
    // Regression tests for #2007872.
    $query = $this->buildSearch(NULL, [], [], FALSE);
    $conditions = $query->createAndAddConditionGroup('OR');
    $conditions->addCondition('id', 3);
    $conditions->addCondition('type', 'article');
    $query->sort('search_api_id', QueryInterface::SORT_DESC);
    $results = $query->execute();
    $this->assertEquals($results->getResultCount(), 3, 'OR filter on field with NULLs');
    $this->assertEquals(
      array_keys(
        $results->getResultItems()
      ),
      $this->getItemIds(
        [5, 4, 3]
      ),
      'OR filter on field with NULLs'
    );
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());

    // Regression tests for #1863672.
    $query = $this->buildSearch();
    $filter = $query->createAndAddConditionGroup('OR');
    $filter->addCondition('keywords', 'orange');
    $filter->addCondition('keywords', 'apple');
    $query->sort('search_api_id', 'ASC');
    $results = $query->execute();
    $this->assertEquals($results->getResultCount(), 4, 'OR filter on multi-valued field returned correct number of results.');
    $this->assertEquals(
      array_keys(
        $results->getResultItems()
      ),
      $this->getItemIds(
        [1, 2, 4, 5]
      ),
      'OR filter on multi-valued field returned correct result.'
    );
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());

    $query = $this->buildSearch();
    $filter = $query->createAndAddConditionGroup('OR');
    $filter->addCondition('keywords', 'orange');
    $filter->addCondition('keywords', 'strawberry');
    $filter = $query->createAndAddConditionGroup('OR');
    $filter->addCondition('keywords', 'apple');
    $filter->addCondition('keywords', 'grape');
    $query->sort('search_api_id', 'ASC');
    $results = $query->execute();
    $this->assertEquals($results->getResultCount(), 3, 'Multiple OR filters on multi-valued field returned correct number of results.');
    $this->assertEquals(
      array_keys(
        $results->getResultItems()
      ),
      $this->getItemIds(
        [2, 4, 5]
      ),
      'Multiple OR filters on multi-valued field returned correct result.'
    );
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());

    $query = $this->buildSearch();
    $filter1 = $query->createAndAddConditionGroup('OR');
    $filter = $query->createConditionGroup('AND');
    $filter->addCondition('keywords', 'orange');
    $filter->addCondition('keywords', 'apple');
    $filter1->addConditionGroup($filter);
    $filter = $query->createConditionGroup('AND');
    $filter->addCondition('keywords', 'strawberry');
    $filter->addCondition('keywords', 'grape');
    $filter1->addConditionGroup($filter);
    $query->sort('search_api_id', 'ASC');
    $results = $query->execute();
    $this->assertEquals($results->getResultCount(), 3, 'Complex nested filters on multi-valued field returned correct number of results.');
    $this->assertEquals(
      array_keys(
        $results->getResultItems()
      ),
      $this->getItemIds(
        [2, 4, 5]
      ),
      'Complex nested filters on multi-valued field returned correct result.'
    );
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());

    // Regression tests for #2111753.
    $keys = [
      '#conjunction' => 'OR',
      0 => 'foo',
      1 => 'test',
    ];
    $query = $this->buildSearch($keys, [], ['name']);
    $query->sort('search_api_id', 'ASC');
    $results = $query->execute();
    $this->assertEquals($results->getResultCount(), 3, 'OR keywords returned correct number of results.');
    $this->assertEquals(
      array_keys(
        $results->getResultItems()
      ),
      $this->getItemIds(
        [1, 2, 4]
      ),
      'OR keywords returned correct result.'
    );
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());

    $query = $this->buildSearch(
      $keys, [], [
        'name',
        'body',
      ]
    );
    $query->range(0, 0);
    $results = $query->execute();
    $this->assertEquals($results->getResultCount(), 5, 'Multi-field OR keywords returned correct number of results.');
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());

    $keys = [
      '#conjunction' => 'OR',
      0 => 'foo',
      1 => 'test',
      2 => [
        '#conjunction' => 'AND',
        0 => 'bar',
        1 => 'baz',
      ],
    ];
    $query = $this->buildSearch($keys, [], ['name']);
    $query->sort('search_api_id', 'ASC');
    $results = $query->execute();
    $this->assertEquals($results->getResultCount(), 4, 'Nested OR keywords returned correct number of results.');
    $this->assertEquals(
      array_keys(
        $results->getResultItems()
      ),
      $this->getItemIds(
        [1, 2, 4, 5]
      ),
      'Nested OR keywords returned correct result.'
    );
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());

    // This test is from BackendTestBase::regressionTest2111753().
    $keys = [
      '#conjunction' => 'OR',
      0 => [
        '#conjunction' => 'AND',
        0 => 'foo',
        1 => 'test',
      ],
      1 => [
        '#conjunction' => 'AND',
        0 => 'bar',
        1 => 'baz',
      ],
    ];
    $query = $this->buildSearch(
      $keys, [], [
        'name',
        'body',
      ]
    );
    $query->sort('search_api_id', 'ASC');
    $results = $query->execute();
    $this->assertEquals($results->getResultCount(), 4, 'Nested multi-field OR keywords returned correct number of results.');
    $this->assertEquals(
      array_keys(
        $results->getResultItems()
      ),
      $this->getItemIds(
        [1, 2, 4, 5]
      ),
      'Nested multi-field OR keywords returned correct result.'
    );
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());

    // Regression tests for #2127001.
    $keys = [
      '#conjunction' => 'AND',
      '#negation' => TRUE,
      0 => 'foo',
      1 => 'bar',
    ];
    $results = $this->buildSearch($keys)
      ->sort('search_api_id', 'ASC')
      ->execute();
    $this->assertEquals($results->getResultCount(), 2, 'Negated AND fulltext search returned correct number of results.');
    $this->assertEquals(
      array_keys($results->getResultItems()), $this->getItemIds(
      [3, 4]
    ), 'Negated AND fulltext search returned correct result.'
    );
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());
  }

  /**
   * Regression Tests 2.
   */
  protected function regressionTests2() {
    // Create a "keywords" field on the test entity type.
    FieldStorageConfig::create(
      [
        'field_name' => 'prices',
        'entity_type' => 'entity_test_mulrev_changed',
        'type' => 'decimal',
        'cardinality' => FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
      ]
    )->save();
    FieldConfig::create(
      [
        'field_name' => 'prices',
        'entity_type' => 'entity_test_mulrev_changed',
        'bundle' => 'item',
        'label' => 'Prices',
      ]
    )->save();

    // Regression test for #1916474.
    /** @var \Drupal\search_api\IndexInterface $index */
    $index = $this->getIndex();
    $this->addField($index, 'prices', 'decimal');
    $success = $index->save();
    $this->assertNotEmpty($success, 'The index field settings were successfully changed.');

    // Reset the static cache so the new values will be available.
    \Drupal::entityTypeManager()
      ->getStorage('search_api_server')
      ->resetCache([$this->serverId]);
    \Drupal::entityTypeManager()
      ->getStorage('search_api_index')
      ->resetCache([$this->serverId]);

    $testEntity = $this->addTestEntity(6, [
      'name' => 'lorem ipsum',
      'body' => 'lorem ipsum',
      'type' => 'item',
      'prices' => ['3.5', '3.25', '3.75', '3.5'],
    ]);
    $this->getIndex()->indexSpecificItems([
      'entity:entity_test_mulrev_changed/6:en' => $testEntity->getTypedData(),
    ]);

    $query = $this->buildSearch(NULL, ['prices,3.25']);
    sleep(1);
    $results = $query->execute();
    $this->assertEquals($results->getResultCount(), 1, 'Filter on decimal field returned correct number of results.');
    $this->assertEquals(array_keys($results->getResultItems()), $this->getItemIds([6]), 'Filter on decimal field returned correct result.');
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());

    $query = $this->buildSearch(NULL, ['prices,3.50']);
    sleep(1);
    $results = $query->execute();
    $this->assertEquals($results->getResultCount(), 1, 'Filter on decimal field returned correct number of results.');
    $this->assertEquals(array_keys($results->getResultItems()), $this->getItemIds([6]), 'Filter on decimal field returned correct result.');
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());
  }

  /**
   * Tests whether some test searches have the correct results.
   */
  protected function searchSuccess() {
    $prepareSearch = $this->buildSearch('test')
      ->range(1, 2)
      ->sort('search_api_id', 'ASC');
    sleep(1);
    $results = $prepareSearch->execute();
    $this->assertEquals($results->getResultCount(), 4, 'Search for »test« returned correct number of results.');
    $this->assertEquals(
      array_keys($results->getResultItems()), $this->getItemIds(
      [
        2,
        3,
      ]
    ), 'Search for »test« returned correct result.'
    );
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());

    $ids = $this->getItemIds([2]);
    $id = reset($ids);
    $this->assertEquals(key($results->getResultItems()), $id);
    $this->assertEquals($results->getResultItems()[$id]->getId(), $id);
    $this->assertEquals($results->getResultItems()[$id]->getDatasourceId(), 'entity:entity_test_mulrev_changed');

    $prepareSearch = $this->buildSearch('test foo')
      ->sort('search_api_id', 'ASC');
    sleep(1);
    $results = $prepareSearch->execute();
    $this->assertEquals($results->getResultCount(), 3, 'Search for »test foo« returned correct number of results.');
    $this->assertEquals(
      array_keys(
        $results->getResultItems()
      ),
      $this->getItemIds(
        [1, 2, 4]
      ),
      'Search for »test foo« returned correct result.'
    );
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());

    $prepareSearch = $this->buildSearch('foo', ['type,item'])
      ->sort('search_api_id', 'ASC');
    sleep(1);
    $results = $prepareSearch->execute();
    $this->assertEquals($results->getResultCount(), 2, 'Search for »foo« returned correct number of results.');
    $this->assertEquals(
      array_keys($results->getResultItems()), $this->getItemIds(
      [
        1,
        2,
      ]
    ), 'Search for »foo« returned correct result.'
    );
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());

    $keys = [
      '#conjunction' => 'AND',
      0 => 'test',
      [
        '#conjunction' => 'OR',
        0 => 'baz',
        1 => 'foobar',
      ],
      [
        '#conjunction' => 'OR',
        '#negation' => TRUE,
        0 => 'bar',
        1 => 'fooblob',
      ],
    ];
    $prepareSearch = $this->buildSearch($keys);
    sleep(1);
    $results = $prepareSearch->execute();
    $this->assertEquals($results->getResultCount(), 1, 'Complex search 1 returned correct number of results.');
    $this->assertEquals(array_keys($results->getResultItems()), $this->getItemIds([4]), 'Complex search 1 returned correct result.');
    $this->assertEmpty($results->getIgnoredSearchKeys());
    $this->assertEmpty($results->getWarnings());
  }

}
