<?php

namespace Drupal\elasticsearch_search_api\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Render\RendererInterface;
use Drupal\elasticsearch_search_api\KeymatchService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class KeymatchForm.
 *
 * @package Drupal\general\Form
 */
class KeymatchForm extends ConfigFormBase {

  const CONFIG_KEY = 'elasticsearch_search_api.keymatch';

  /**
   * The config instance.
   *
   * @var \Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig
   */
  protected $configInstance;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The keymatch service.
   *
   * @var \Drupal\elasticsearch_search_api\KeymatchService
   */
  protected $keymatchService;

  /**
   * KeymatchForm constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config instance.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   * @param \Drupal\elasticsearch_search_api\KeymatchService $keymatch_service
   *   The keymatch service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typedConfigManager, RendererInterface $renderer, KeymatchService $keymatch_service) {
    parent::__construct($config_factory, $typedConfigManager);

    $this->configInstance = $this->config(self::CONFIG_KEY);
    $this->renderer = $renderer;
    $this->keymatchService = $keymatch_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('renderer'),
      $container->get('elasticsearch_search_api.keymatch_service')
    );
  }

  /**
   * Implements \Drupal\Core\Form\FormInterface::getFormID().
   */
  public function getFormId() {
    return 'elasticsearch_search_api_keymatch_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      self::CONFIG_KEY,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['keymatch_fieldset'] = [
      '#type' => 'fieldset',
      '#collapsible' => FALSE,
      '#collapsed' => FALSE,
      '#title' => '',
    ];

    $form['keymatch_fieldset']['keymatches'] = [
      '#title' => t('Keymatches'),
      '#type' => 'textarea',
      '#default_value' => $this->configInstance->get('keymatches'),
      '#rows' => 15,
    ];

    $form['keymatch_fieldset']['keymatch_information'] = [
      '#markup' => $this->getKeymatchInfo(),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Provide help text for configuring keymatches.
   *
   * @return string
   *   Info message markup.
   *
   * @throws \Exception
   */
  protected function getKeymatchInfo() {
    $types_list = [
      '#theme' => 'item_list',
      '#items' => [
        KeymatchService::SEARCH_KEYMATCH_TYPE_TERM . ' (= ' . t('All terms, space delimited, occur anywhere in search query. Case-insensitive.') . ')',
        KeymatchService::SEARCH_KEYMATCH_TYPE_PHRASE . ' (= ' . t('Phrase occurs anywhere in search query. Case-insensitive.') . ')',
        KeymatchService::SEARCH_KEYMATCH_TYPE_EXACT . ' (= ' . t('Phrase/Term matches search query exactly. Case-sensitive.') . ')',
      ],
    ];

    return '<div class="instruction-msg">' . t('Enter 1 entry per line (comma separated)') . ':<br />' .
      '- ' . t('Search term') . ',' . t('{Type}') . ',' . t('URL for match') . ',' . t('Title for match') . '<br />' .
      '- ' . 'Example,TERM,https://www.example.be/test,Example Keymatch <br />' .
      '- ' . 'Second Example,TERM,https://www.example.be/test,Second Keymatch <br />' .
      '- ' . 'Third Example,EXACT,https://www.example.be/test,Third Keymatch </div><br />' .
      '<div class="instruction-tokens">' . t('{Type} can be one of the following') . ':' . $this->renderer->render($types_list) . '</div>';
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $config = $this->keymatchService->getKeymatchConfiguration($form_state->getValue('keymatches'));

    $errors = '';
    foreach ($config as $keymatch_entry) {
      if (!$this->keymatchService->isValid($keymatch_entry, TRUE)) {
        $errors .= '<li>Invalid entry: ' . $keymatch_entry . "</li>";
      }
    }
    $errors = empty($errors) ? $errors : '<ul>' . $errors . '</ul>';
    $errors = Markup::create($errors);

    if (!empty($errors)) {
      $message = $this->t('Not all provided keymatches were valid. @errors', ['@errors' => $errors]);
      $form_state->setErrorByName('keymatches', $message);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    foreach ($form_state->cleanValues()->getValues() as $key => $formValue) {
      $this->configInstance->set($key, $formValue);
    }

    $this->configInstance->save();

    parent::submitForm($form, $form_state);
  }

}
