<?php

namespace Drupal\email_obfuscator;

/**
 * Service to obfuscate email addresses in content.
 */
class EmailObfuscatorService {

  /**
   * Obfuscate email addresses in the given content.
   *
   * Revert emails in mailto-links and add a display-none-span in the
   * email-texts so bots can't read them - hopefully.
   *
   * @param string $content
   *   The content to process, which may contain email addresses.
   * @param bool $useDataNoSnippet
   *   Whether to use the data-nosnippet attribute.
   *
   * @return string
   *   The processed string with obfuscated email addresses.
   *
   * @throws \Exception
   *
   * @see https://web.archive.org/web/20180908103745/http://techblog.tilllate.com/2008/07/20/ten-methods-to-obfuscate-e-mail-addresses-compared/
   * @see http://jasonpriem.com/obfuscation-decoder/
   */
  public function obfuscateEmails(string $content, bool $useDataNoSnippet = TRUE): string {
    $obfuscateMailtoLinks = $this->obfuscateMailtoLinks($content);

    // @todo Use random string for displayNoneText.
    return $this->obfuscateEmailStrings($obfuscateMailtoLinks, "zilch", $useDataNoSnippet);
  }

  /**
   * Obfuscate mailto links in the given content.
   *
   * Find the mailto links, revert them and add onclick which reverts
   * mailto-link back to normal. But only on the first click. Invalid emails
   * are ignored.
   *
   * @param string $content
   *   The content to process, which may contain mailto links.
   *
   * @return string
   *   The processed string with obfuscated mailto links.
   *
   * @throws \Drupal\email_obfuscator\EmailObfuscatorException
   */
  private function obfuscateMailtoLinks(string $content): string {
    $mailtoRegex = '/(href=)"mailto:([^"]+)"/';

    return preg_replace_callback(
      $mailtoRegex,
      function ($matches) {
        // If the email is invalid, don't do anything.
        if (!filter_var($matches[2], FILTER_VALIDATE_EMAIL)) {
          return $matches[0];
        }

        // The dataset.obfuscated property is used to check if the link has
        // already been reverted.
        $jsString = "!this.dataset.obfuscated && (this.dataset.obfuscated = true) && this.setAttribute('href', 'mailto:' + this.getAttribute('href').substring(7).split('').reverse().join(''))";

        // Use onmousedown and on focus to cover all cases: right-click,
        // left-click and select with tab.
        // The onfocus event would do it for most browsers, but Safari needs
        // onmousedown.
        return $matches[1] . "\"mailto:" . strrev(
          $matches[2]
        ) . "\" onfocus=\"" . $jsString . "\" onmousedown=\"" . $jsString . "\"";
      },
      $content
    ) ?? throw new EmailObfuscatorException(
      'Removing mailtos with regex and adding onclick to email links failed.'
    );
  }

  /**
   * Obfuscate email strings in the given content.
   *
   * Get all email strings that are not in an HTML element and add a
   * display-none-span in the middle of the string.
   * Invalid emails are ignored.
   *
   * @param string $content
   *   The content to process, which may contain email addresses.
   * @param string $displayNoneText
   *   The text to be used in the display-none-span. Defaults to "zilch".
   * @param bool $useDataNoSnippet
   *   The data-nosnippet attribute to prevent the email from being indexed by
   *   search engines. Defaults to TRUE.
   *
   * @return string
   *   The processed string with email addresses obfuscated.
   *
   * @throws \Exception
   */
  private function obfuscateEmailStrings(string $content, string $displayNoneText, bool $useDataNoSnippet = TRUE): string {
    // Get all email strings that are not in an html element.
    $emailRegex = '/(<[^>]+)|(([\w\-\.]+@)([\w\-\.]+\.[a-zA-Z]{2,}))/';

    // Exclamation marks are invalid in emails. we use them as delimiters, so
    // we don't replace unwanted parts of the email.
    $stringToReplace = "!" . $displayNoneText . "!";

    // If the data-nosnippet attribute should be used, add it to the span.
    $dataNoSnippetString = $useDataNoSnippet ? "data-nosnippet" : "";

    return preg_replace_callback(
      $emailRegex,
      function ($matches) use ($stringToReplace, $dataNoSnippetString) {
        // If the email is in an HTML element or if the email is invalid, don't
        // do anything.
        if (!empty($matches[1]) || !filter_var($matches[2], FILTER_VALIDATE_EMAIL)) {
          return $matches[0];
        }

        // Otherwise add the display-none span.
        return $matches[3] . "<span style='display:none' $dataNoSnippetString>" . $stringToReplace . "</span>" . $matches[4];
      },
      $content
    ) ?? throw new \Exception('Adding display-none-span failed.');
  }

}
