<?php

declare(strict_types=1);

namespace Drupal\email_obfuscator\EventSubscriber;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Site\Settings;
use Drupal\email_obfuscator\EmailObfuscatorException;
use Drupal\email_obfuscator\EmailObfuscatorService;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber to obfuscate email addresses in responses.
 *
 * This subscriber listens to the kernel response event and applies a filter
 * to obfuscate email addresses in the response content.
 *
 * @package Drupal\email_obfuscator\EventSubscriber
 */
final class EmailObfuscatorResponseFilter implements EventSubscriberInterface {

  /**
   * Constructs an EmailObfuscatorSubscriber object.
   */
  public function __construct(
    private readonly EmailObfuscatorService $emailObfuscator,
    private readonly AdminContext $adminContext,
    private readonly LoggerChannelInterface $logger,
  ) {}

  /**
   * Kernel response event handler.
   */
  public function onKernelResponse(ResponseEvent $event): void {
    $request = $event->getRequest();
    $response = $event->getResponse();

    $routeObject = $request->attributes->get('_route_object');
    $route = $request->attributes->get('_route');
    $content = $response->getContent();
    // If there is no route object, no route or no content, do nothing.
    if (!$routeObject || !$route || empty($content)) {
      return;
    }

    $isAdminPath = $this->adminContext->isAdminRoute($routeObject);
    $ignoredRoutes = Settings::get('email_obfuscator')['ignored_routes'] ?? [];
    $isRouteIgnored = in_array($route, $ignoredRoutes);
    // Check whether an Ajax route with Webform content has been sent;
    // this content should not be obfuscated.
    $isAjaxRequest = $request->isXmlHttpRequest();
    $isWebForm = $request->request->has('form_id') && str_starts_with($request->request->get('form_id'), 'webform');

    // Don't obfuscate emails in backoffice or on ignored routes or
    // from webforms with enabled ajax.
    if ($isAdminPath
      || $isRouteIgnored
      || ($isAjaxRequest && $isWebForm)) {
      return;
    }

    try {
      $useDataNoSnippet = Settings::get('email_obfuscator')['use_datanosnippet'] ?? TRUE;
      $obfuscateEmails = $this->emailObfuscator->obfuscateEmails($content, $useDataNoSnippet);
      $response->setContent($obfuscateEmails);
      $event->setResponse($response);
    }
    catch (EmailObfuscatorException $e) {
      $this->logger->error($e->getMessage());
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::RESPONSE => ['onKernelResponse'],
    ];
  }

}
