<?php

namespace Drupal\email_otp_login\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\email_otp_login\Mailer;
use Drupal\email_otp_login\OtpGenerator;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 *
 */
class OtpEmail extends FormBase {

  protected $otpMailer;
  protected $otpGenerator;

  public function __construct(Mailer $mailer, OtpGenerator $otpGenerator) {
    $this->otpMailer = $mailer;
    $this->otpGenerator = $otpGenerator;
  }

  /**
   *
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('email_otp_login.mailer'),
      $container->get('email_otp_login.generator')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'otp_email_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Check if the user is already logged in.
    if (\Drupal::currentUser()->isAuthenticated()) {
      \Drupal::messenger()->addMessage($this->t('You are already logged in. You do not need to enter an OTP.'));
      $form['#title'] = $this->t('You are already logged in.');

      // Return an empty form as the user doesn't need to proceed with OTP.
      return $form;
    }

    // Collect email address from user.
    $form['email'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Enter Your Email Address'),
      '#description' => $this->t('We will send an OTP to this email address.'),
      '#required' => TRUE,
    ];

    // Submit button.
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit Email'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $email = $form_state->getValue('email');

    // Check if email is valid.
    if (empty($email)) {
      \Drupal::messenger()->addError($this->t('Please provide a valid email address.'));
      return;
    }

    // Generate OTP and send it.
    $otp = $this->otpGenerator->generateOtp();
    $this->otpMailer->sendOtpEmail($email, $otp);

    // Store OTP temporarily in session or state.
    \Drupal::state()->set('otp_code_' . $email, $otp);

    // Redirect user to OTP validation form.
    \Drupal::messenger()->addMessage($this->t('OTP has been sent to your email. Please check your inbox.'));
    $form_state->setRedirect('email_otp_login.validate_otp', ['email' => $email]);
  }

}
