<?php

namespace Drupal\email_otp_login\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\user\Entity\User;

class OtpValidation extends FormBase {

  /**
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  public function __construct(RouteMatchInterface $route_match, StateInterface $state, MessengerInterface $messenger) {
    $this->routeMatch = $route_match;
    $this->state = $state;
    $this->messenger = $messenger;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_route_match'),
      $container->get('state'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'otp_validation_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Get the email passed via URL.
    $email = $this->routeMatch->getParameter('email');

    $form['otp'] = [
      '#type' => 'password',
      '#title' => $this->t('Enter OTP'),
      '#description' => $this->t('Please enter the OTP sent to your email.'),
      '#required' => TRUE,
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit OTP'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $email = $this->routeMatch->getParameter('email');
    $otp_entered = $form_state->getValue('otp');

    // Get OTP stored in state.
    $stored_otp = $this->state->get('otp_code_' . $email);

    if ($otp_entered == $stored_otp) {
      // OTP is valid, log in the user.
      $user = user_load_by_mail($email);
      if ($user) {
        user_login_finalize($user);
        $this->state->delete('otp_code_' . $email);

        $this->messenger->addMessage($this->t('OTP validated successfully. You are now logged in.'));

        $profile_url = Url::fromRoute('entity.user.canonical', ['user' => $user->id()]);
        $form_state->setRedirectUrl($profile_url);
      }
      else {
        $this->messenger->addError($this->t('No user found for this email.'));
      }
    }
    else {
      $this->messenger->addError($this->t('Invalid OTP.'));
    }
  }

}
