<?php

namespace Drupal\email_otp_login\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\email_otp_login\MailerService;
use Drupal\email_otp_login\OtpGeneratorService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\State\State;

/**
 *
 */
class OtpEmailForm extends FormBase {

  protected $otpMailer;
  protected $otpGenerator;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;
  
  /**
   * The state.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   *
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   Current user.
   * @param Drupal\Core\State\State $state
   *   The object State.
   */
  public function __construct(MailerService $mailer, OtpGeneratorService $otpGenerator, MessengerInterface $messenger, AccountInterface $current_user, State $state) {
    $this->otpMailer = $mailer;
    $this->otpGenerator = $otpGenerator;
    $this->messenger = $messenger;
    $this->currentUser = $current_user;
    $this->state = $state;
  }

  /**
   *
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('email_otp_login.mailer'),
      $container->get('email_otp_login.generator'),
      $container->get('messenger'),
      $container->get('current_user'),
      $container->get('state')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'otp_email_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Check if the user is already logged in.
    if ($this->currentUser->isAuthenticated()) {
      $this->messenger->addMessage($this->t('You are already logged in. You do not need to enter an OTP.'));
      $form['#title'] = $this->t('You are already logged in.');

      // Return an empty form as the user doesn't need to proceed with OTP.
      return $form;
    }

    // Collect email address from user.
    $form['email'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Enter Your Email Address'),
      '#description' => $this->t('We will send an OTP to this email address.'),
      '#required' => TRUE,
    ];

    // Submit button.
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit Email'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $email = $form_state->getValue('email');

    // Check if email is valid.
    if (empty($email)) {
      $this->messenger->addError($this->t('Please provide a valid email address.'));
      return;
    }

    // Load user by email.
    $user = user_load_by_mail($email);

    if (!$user) {
      $this->messenger->addError($this->t('Email address not found.'));
      return;
    }

    // Check if user account is blocked.
    if ($user->isBlocked()) {
      $this->messenger->addError(
        $this->t('Your account is blocked. OTP cannot be sent. Please contact the site administrator.')
      );
      return;
    }

    // Generate OTP and send it.
    $otp = $this->otpGenerator->generateOtp();
    $this->otpMailer->sendOtpEmail($email, $otp);

    // Store OTP temporarily.
    $this->state->set('otp_code_' . $email, $otp);

    $this->messenger->addMessage(
      $this->t('OTP has been sent to your email. Please check your inbox.')
    );

    // Redirect to OTP validation form.
    $form_state->setRedirect('email_otp_login.validate_otp', ['email' => $email]);
  }

}
