<?php

namespace Drupal\emedia_library\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'emedia_library_field_image' field type.
 *
 * @FieldType(
 *   id = "emedia_library_field_image",
 *   label = @Translation("eMedia Image"),
 *   description = @Translation("Pulls media assets from eMedia Library."),
 *   default_widget = "emedia_library_widget",
 *   default_formatter = "emedia_library_formatter"
 * )
 */
class EmediaLibraryFieldImage extends FieldItemBase {

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties['asset_id'] = DataDefinition::create('string')
      ->setLabel(t('Asset ID'));

    $properties['preset_id'] = DataDefinition::create('string')
      ->setLabel(t('Preset Id'))
      ->setDescription(t('Select the Preset Id for this asset.'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    return [
      'columns' => [
        'asset_id' => [
          'type' => 'varchar',
          'length' => 255,
        ],
        'preset_id' => [
          'type' => 'varchar',
          'length' => 128,
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    return empty($this->get('asset_id')->getValue());
  }

  /**
   * Define default settings for the field type.
   *
   * @return array
   *   An array of default settings.
   */
  public static function defaultFieldSettings() {
    return [
      'preset_id' => 'webplargeimage',
    ];
  }

  /**
   * Build the settings form for the field type.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return array
   *   The modified form array.
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state) {
    $settings = $this->getSettings();
  
    // Retrieve  comma-separated list from the module settings.
    $options = $this->fetchImageSizeOptions();
  
    $form2['preset_id'] = [
      '#type' => 'select',
      '#title' => t('Preset Id'),
      '#options' => $options,
      '#default_value' => $settings['preset_id'],
      '#description' => t('Select the default Preset Id for this field.'),
    ];
  
    return $form2;
  }


public static function fetchImageSizeOptions() {
  $options = [];

  // Get the eMedia Library URL and API key from the module settings.
  $emedialibraryUrl = \Drupal::config('emedia_library.settings')->get('emedialibrary-url');
  $entermediaKey = \Drupal::config('emedia_library.settings')->get('emedialibrary-key');
  $mediadbUrl = $emedialibraryUrl . "/mediadb/services/lists/search/convertpreset";

 $query = [
    "page" => "1",
    "hitsperpage" => "40",
    "query" => [
      "terms" => [
        [
          "field" => "id",
          "operation" => "matches",
          "value" => "*"
        ]
      ]
    ]
  ];

  try {
    $client = \Drupal::httpClient();
    $response = $client->post($mediadbUrl, [
      'headers' => [
        'Content-Type' => 'application/json',
        'X-tokentype' => 'entermedia', 
        'X-token' => $entermediaKey,
      ],
      'body' => json_encode($query),
      'timeout' => 3,
    ]);
    $httpcode = $response->getStatusCode();

    if ($httpcode >= 200 && $httpcode < 300 && !empty($response)) {
      $body = $response->getBody()->getContents();
      $json = json_decode($body, TRUE);
      if (isset($json['results']) && is_array($json['results'])) {
        foreach ($json['results'] as $preset) {
          $options[$preset["id"]] = $preset["name"];
        }
      }
    }
  }
  catch (RequestException $e) {
    \Drupal::logger('emedia_library')->error('API request failed: @message', ['@message' => $e->getMessage()]);
  }

  return $options;
}

  /**
   * Validate the settings form for the field type.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function validateFieldSettingsForm(array &$form, FormStateInterface $form_state) {
    // Add validation logic if needed.
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $settings = $this->getSettings();

    $summary[] = t('Default Preset Id: @size', ['@size' => $settings['preset_id']]);

    return $summary;
  }
}