<?php

declare(strict_types=1);

namespace Drupal\empty_headings_filter\Plugin\Filter;

use Drupal\Component\Utility\Html;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\filter\Attribute\Filter;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;
use Drupal\filter\Plugin\FilterInterface;

/**
 * Removes empty headings by replacing them with paragraph tags.
 *
 * This filter scans h1–h6 and, when a heading has no visible text content,
 * replaces it with a <p> while preserving original attributes and any children.
 */
#[Filter(
  id: "empty_headings_filter",
  title: new TranslatableMarkup("Empty headings filter"),
  description: new TranslatableMarkup("Replaces empty h1–h6 elements with p tags, preserving attributes and children."),
  type: FilterInterface::TYPE_TRANSFORM_REVERSIBLE,
  weight: 10
)]
final class EmptyHeadingsFilter extends FilterBase {

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode): FilterProcessResult {
    if (empty($text)) {
      return new FilterProcessResult((string) $text);
    }

    $dom = Html::load($text);
    $xpath = new \DOMXPath($dom);

    $nodeList = $xpath->query('//h1|//h2|//h3|//h4|//h5|//h6');

    if ($nodeList === FALSE || $nodeList->length === 0) {
      return new FilterProcessResult($text);
    }

    // Copy to array to avoid mutating a live NodeList during replacements.
    $headings = [];
    foreach ($nodeList as $el) {
      $headings[] = $el;
    }

    foreach ($headings as $heading) {
      if (!($heading instanceof \DOMElement)) {
        continue;
      }

      if ($this->isHeadingEmpty($heading)) {
        $this->replaceHeadingWithParagraph($dom, $heading);
      }
    }

    $output = Html::serialize($dom);
    return new FilterProcessResult($output);
  }

  /**
   * Determine if a heading element has any visible text content.
   *
   * Rules:
   * - Strip comments before measuring text.
   * - Treat &nbsp; as whitespace.
   * - Collapses whitespace and trims.
   */
  private function isHeadingEmpty(\DOMElement $heading): bool {
    // Clone so we can strip comments without touching the live DOM.
    $clone = $heading->cloneNode(TRUE);
    \assert($clone instanceof \DOMElement);

    $this->removeComments($clone);

    // Normalize text content.
    $text = $clone->textContent ?? '';
    // Decode HTML entities and replace NBSP with regular spaces.
    $text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5);
    $text = preg_replace('/\x{00A0}/u', ' ', $text) ?? $text;
    // Collapse whitespace and trim.
    $text = trim(preg_replace('/\s+/u', ' ', $text) ?? '');

    return $text === '';
  }

  /**
   * Replace a heading with a <p>, preserving attributes and children.
   */
  private function replaceHeadingWithParagraph(\DOMDocument $dom, \DOMElement $heading): void {
    $p = $dom->createElement('p');

    // Preserve all attributes from the heading.
    if ($heading->hasAttributes()) {
      /** @var \DOMAttr $attr */
      foreach ($heading->attributes as $attr) {
        $p->setAttribute($attr->nodeName, $attr->nodeValue ?? '');
      }
    }

    // Move all children to the new <p>.
    while ($heading->firstChild) {
      $p->appendChild($heading->firstChild);
    }

    // Replace node in DOM.
    $heading->parentNode?->replaceChild($p, $heading);
  }

  /**
   * Remove all comment nodes from a subtree (in place).
   */
  private function removeComments(\DOMNode $node): void {
    for ($child = $node->firstChild; $child !== NULL; $child = $child->nextSibling) {
      if ($child->nodeType === XML_COMMENT_NODE) {
        $node->removeChild($child);
        // Restart from the beginning since we've mutated siblings.
        $this->removeComments($node);
        return;
      }
      if ($child->hasChildNodes()) {
        $this->removeComments($child);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function tips($long = FALSE) {
    return (string) $this->t('Converts empty headings (H1–H6) into paragraphs to avoid accessibility issues while keeping layout spacing.');
  }

}
