<?php

declare(strict_types=1);

namespace Drupal\enforce_revision\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\enforce_revision\Service\RevisionEnforcer;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Enforce Revision settings.
 */
final class EnforceRevisionSettingsForm extends ConfigFormBase {

  /**
   * The revision enforcer service.
   */
  private RevisionEnforcer $revisionEnforcer;

  /**
   * Constructs a new EnforceRevisionSettingsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\enforce_revision\Service\RevisionEnforcer $revision_enforcer
   *   The revision enforcer service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    RevisionEnforcer $revision_enforcer
  ) {
    parent::__construct($config_factory);
    $this->revisionEnforcer = $revision_enforcer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('enforce_revision.revision_enforcer')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['enforce_revision.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'enforce_revision_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config          = $this->config('enforce_revision.settings');
    $enabled_bundles = $config->get('enabled_bundles') ?: [];
    $bundles         = $this->revisionEnforcer->getAvailableBundles();
    $roles           = $this->revisionEnforcer->getAvailableRoles();

    $form['description'] = [
      '#type'   => 'markup',
      '#markup' => '<p>' . $this->t('Configure which content types should enforce revision creation for specific user roles. When enabled, users with the selected roles will not be able to turn off the "Create new revision" option when editing content of the specified type.') . '</p>',
    ];

    $form['bundle_configurations'] = [
      '#type'        => 'vertical_tabs',
      '#title'       => $this->t('Content Type Configurations'),
      '#description' => $this->t('Configure revision enforcement for each content type.'),
    ];

    // Create a tab for each content type.
    foreach ($bundles as $bundle_id => $bundle_label) {
      $form['bundle_' . $bundle_id] = [
        '#type'  => 'details',
        '#title' => $bundle_label,
        '#group' => 'bundle_configurations',
      ];

      // Get current configuration for this bundle.
      $current_roles         = [];
      $current_hide_checkbox = FALSE;
      foreach ($enabled_bundles as $bundle_config) {
        if ($bundle_config['bundle'] === $bundle_id) {
          $current_roles         = $bundle_config['roles'] ?? [];
          $current_hide_checkbox = $bundle_config['hide_checkbox'] ?? FALSE;
          break;
        }
      }

      $form['bundle_' . $bundle_id]['enabled_' . $bundle_id] = [
        '#type'          => 'checkbox',
        '#title'         => $this->t('Enable revision enforcement for @bundle', ['@bundle' => $bundle_label]),
        '#default_value' => !empty($current_roles),
        '#description'   => $this->t('Check this to enforce revision creation for this content type.'),
      ];

      $form['bundle_' . $bundle_id]['roles_' . $bundle_id] = [
        '#type'          => 'checkboxes',
        '#title'         => $this->t('Restricted roles'),
        '#options'       => $roles,
        '#default_value' => $current_roles,
        '#description'   => $this->t('Select which roles should be restricted from disabling revision creation for this content type.'),
        '#states'        => [
          'visible' => [
            ':input[name="enabled_' . $bundle_id . '"]' => ['checked' => TRUE],
          ],
          'required' => [
            ':input[name="enabled_' . $bundle_id . '"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form['bundle_' . $bundle_id]['hide_checkbox_' . $bundle_id] = [
        '#type'          => 'checkbox',
        '#title'         => $this->t('Hide revision checkbox entirely'),
        '#default_value' => $current_hide_checkbox,
        '#description'   => $this->t('If checked, the revision checkbox will be completely hidden from the form instead of being disabled. Revisions will be automatically created.'),
        '#states'        => [
          'visible' => [
            ':input[name="enabled_' . $bundle_id . '"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $bundles = $this->revisionEnforcer->getAvailableBundles();

    foreach ($bundles as $bundle_id => $bundle_label) {
      $enabled = $form_state->getValue('enabled_' . $bundle_id);
      $roles   = $form_state->getValue('roles_' . $bundle_id);

      // If enabled but no roles selected, show error.
      if ($enabled && empty(array_filter($roles))) {
        $form_state->setErrorByName(
          'roles_' . $bundle_id,
          $this->t('You must select at least one role when enabling revision enforcement for @bundle.', [
            '@bundle' => $bundle_label,
          ])
        );
      }
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config           = $this->config('enforce_revision.settings');
    $bundles          = $this->revisionEnforcer->getAvailableBundles();
    $enabled_bundles  = [];

    foreach ($bundles as $bundle_id => $bundle_label) {
      $enabled       = $form_state->getValue('enabled_' . $bundle_id);
      $roles         = $form_state->getValue('roles_' . $bundle_id);
      $hide_checkbox = $form_state->getValue('hide_checkbox_' . $bundle_id);

      // Only store configuration for enabled bundles.
      if ($enabled && !empty(array_filter($roles))) {
        $enabled_bundles[] = [
          'bundle'        => $bundle_id,
          'roles'         => array_values(array_filter($roles)),
          'hide_checkbox' => (bool) $hide_checkbox,
        ];
      }
    }

    $config->set('enabled_bundles', $enabled_bundles);
    $config->save();

    $this->messenger()->addStatus($this->t('The configuration options have been saved.'));

    parent::submitForm($form, $form_state);
  }

}
