<?php

declare(strict_types=1);

namespace Drupal\enforce_revision\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\node\NodeInterface;

/**
 * Service for enforcing revision creation on content types.
 */
final class RevisionEnforcer
{

  /**
   * The config factory service.
   */
  private ConfigFactoryInterface $configFactory;

  /**
   * The current user service.
   */
  private AccountProxyInterface $currentUser;

  /**
   * The entity type manager service.
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a new RevisionEnforcer service.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    AccountProxyInterface $current_user,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    $this->configFactory    = $config_factory;
    $this->currentUser      = $current_user;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Determines if revision enforcement is active for a given bundle and user.
   *
   * @param string $bundle
   *   The content type bundle machine name.
   * @param \Drupal\Core\Session\AccountProxyInterface|null $account
   *   The user account to check. Defaults to current user.
   *
   * @return bool
   *   TRUE if revision should be enforced, FALSE otherwise.
   */
  public function isRevisionEnforced(string $bundle, ?AccountProxyInterface $account = NULL): bool
  {
    if ($account === NULL) {
      $account = $this->currentUser;
    }

    // UID 1 is always exempt from revision enforcement.
    if ($account->id() == 1) {
      return FALSE;
    }

    $config         = $this->configFactory->get('enforce_revision.settings');
    $enabled_bundles = $config->get('enabled_bundles') ?: [];
    $user_roles     = $account->getRoles();

    foreach ($enabled_bundles as $bundle_config) {
      if ($bundle_config['bundle'] === $bundle) {
        $restricted_roles = $bundle_config['roles'] ?? [];

        // Check if any of the user's roles are in the restricted list.
        foreach ($user_roles as $role) {
          if (in_array($role, $restricted_roles, TRUE)) {
            return TRUE;
          }
        }
      }
    }

    return FALSE;
  }

  /**
   * Determines if the revision checkbox should be hidden for a bundle.
   *
   * @param string $bundle
   *   The content type bundle machine name.
   * @param \Drupal\Core\Session\AccountProxyInterface|null $account
   *   The user account to check. Defaults to current user.
   *
   * @return bool
   *   TRUE if checkbox should be hidden, FALSE otherwise.
   */
  public function shouldHideRevisionCheckbox(string $bundle, ?AccountProxyInterface $account = NULL): bool
  {
    if ($account === NULL) {
      $account = $this->currentUser;
    }

    // UID 1 is always exempt from revision enforcement and hiding.
    if ($account->id() == 1) {
      return FALSE;
    }

    $config         = $this->configFactory->get('enforce_revision.settings');
    $enabled_bundles = $config->get('enabled_bundles') ?: [];
    $user_roles     = $account->getRoles();

    foreach ($enabled_bundles as $bundle_config) {
      if ($bundle_config['bundle'] === $bundle) {
        $restricted_roles = $bundle_config['roles'] ?? [];
        $hide_checkbox   = $bundle_config['hide_checkbox'] ?? FALSE;

        // Only hide if configured to hide AND user has a restricted role.
        if ($hide_checkbox) {
          foreach ($user_roles as $role) {
            if (in_array($role, $restricted_roles, TRUE)) {
              return TRUE;
            }
          }
        }
      }
    }

    return FALSE;
  }

  /**
   * Gets all available content type bundles.
   *
   * @return array
   *   An array of bundle machine names keyed by bundle machine name.
   */
  public function getAvailableBundles(): array
  {
    $bundles = [];

    try {
      $node_types = $this->entityTypeManager
        ->getStorage('node_type')
        ->loadMultiple();

      foreach ($node_types as $node_type) {
        $bundles[$node_type->id()] = $node_type->label();
      }
    } catch (\Exception $e) {
      // Log error or handle gracefully.
      $bundles = [];
    }

    return $bundles;
  }

  /**
   * Gets all available user roles.
   *
   * @return array
   *   An array of role labels keyed by role machine name.
   */
  public function getAvailableRoles(): array
  {
    $roles = [];

    try {
      $role_entities = $this->entityTypeManager
        ->getStorage('user_role')
        ->loadMultiple();

      foreach ($role_entities as $role) {
        // Exclude anonymous users as they can't edit content.
        if ($role->id() !== 'anonymous') {
          $roles[$role->id()] = $role->label();
        }
      }
    } catch (\Exception $e) {
      // Log error or handle gracefully.
      $roles = [];
    }

    return $roles;
  }

  /**
   * Enforces revision creation on a node form.
   *
   * @param array $form
   *   The form array to modify.
   * @param string $bundle
   *   The content type bundle.
   */
  public function enforceRevisionOnForm(array &$form, string $bundle): void
  {
    // Find the revision field in the form - it could be in different locations.
    $revision_field_path = $this->findRevisionField($form);

    if (empty($revision_field_path)) {
      // Silently return if revision field is not found.
      return;
    }

    // Check if checkbox should be hidden entirely.
    if ($this->shouldHideRevisionCheckbox($bundle)) {
      $revision_field = &$this->getNestedFormElement($form, $revision_field_path);
      if ($revision_field !== NULL) {
        $revision_field['#access'] = FALSE;

        // Find the container to add hidden field and explanation
        $container_path = array_slice($revision_field_path, 0, -1);
        $container = &$this->getNestedFormElement($form, $container_path);

        if ($container !== NULL) {
          // Also add a hidden field to ensure revision is created.
          $container['revision_hidden'] = [
            '#type'  => 'hidden',
            '#value' => 1,
          ];
          // Add explanatory text where the checkbox would be.
          $container['revision_explanation'] = [
            '#type'   => 'markup',
            '#markup' => '<div class="form-item"><strong>' .
              $this->t('New revisions are automatically created for this content type.') .
              '</strong></div>',
            '#weight' => ($revision_field['#weight'] ?? 0) + 1,
          ];
        }
      }
    } elseif ($this->isRevisionEnforced($bundle)) {
      // Force the revision checkbox to be checked and disabled.
      $revision_field = &$this->getNestedFormElement($form, $revision_field_path);
      if ($revision_field !== NULL) {
        $revision_field['#default_value'] = TRUE;
        $revision_field['#disabled'] = TRUE;
        $revision_field['#description'] = $this->t('Revision creation is enforced for this content type and your role.');
      }
    }
  }

  /**
   * Finds the revision field path in a form array.
   *
   * @param array $form
   *   The form array to search.
   *
   * @return array
   *   The path to the revision field as an array of keys, or empty array if not found.
   */
  private function findRevisionField(array $form): array
  {
    // Common paths where the revision field might be located
    $possible_paths = [
      ['revision_information', 'revision'],
      ['revision'],
      ['meta', 'revision'],
      ['options', 'revision'],
      ['advanced', 'revision_information', 'revision'],
    ];

    foreach ($possible_paths as $path) {
      if ($this->getNestedFormElement($form, $path) !== NULL) {
        return $path;
      }
    }

    // If not found in common paths, search recursively
    return $this->searchRevisionFieldRecursively($form, []);
  }

  /**
   * Recursively search for revision field in form.
   *
   * @param array $element
   *   The form element to search.
   * @param array $current_path
   *   The current path in the form.
   *
   * @return array
   *   The path to the revision field or empty array if not found.
   */
  private function searchRevisionFieldRecursively(array $element, array $current_path): array
  {
    foreach ($element as $key => $value) {
      // Skip system keys
      if (is_string($key) && strpos($key, '#') === 0) {
        continue;
      }

      $new_path = array_merge($current_path, [$key]);

      // Check if this is the revision field
      if ($key === 'revision' && is_array($value) && isset($value['#type']) && $value['#type'] === 'checkbox') {
        return $new_path;
      }

      // Recursively search in sub-elements
      if (is_array($value)) {
        $result = $this->searchRevisionFieldRecursively($value, $new_path);
        if (!empty($result)) {
          return $result;
        }
      }
    }

    return [];
  }

  /**
   * Gets a nested form element by path.
   *
   * @param array $form
   *   The form array.
   * @param array $path
   *   The path to the element as an array of keys.
   *
   * @return array|null
   *   The form element or NULL if not found.
   */
  private function &getNestedFormElement(array &$form, array $path): ?array
  {
    $element = &$form;

    foreach ($path as $key) {
      if (!isset($element[$key])) {
        $null = NULL;
        return $null;
      }
      $element = &$element[$key];
    }

    return $element;
  }

  /**
   * Translates a string.
   *
   * @param string $string
   *   The string to translate.
   * @param array $args
   *   Translation arguments.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The translated string.
   */
  private function t(string $string, array $args = [])
  {
    return \Drupal::translation()->translate($string, $args);
  }
}
