<?php

namespace Drupal\menu_revisions\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Database\Connection;

class MenuRevisionAdvancedController extends ControllerBase {

  protected $database;

  public function __construct(Connection $database) {
    $this->database = $database;
  }

  public static function create(ContainerInterface $container) {
    return new static($container->get('database'));
  }

  // Add this method inside the MenuRevisionController class
  public function revisionPreview($menu, $menu_revision_id) {
    // Render the menu from the revision.
    \Drupal::service('page_cache_kill_switch')->trigger();

    // Sample content (can be a node or static markup for demonstration).
    $sample_content = [
      '#type' => 'container',
      '#attached' => [
        'library' => ['menu_revisions/menu_preview'],
      ],
      '#attributes' => [
        'class' => ['sample-content'],
      ],
      'title' => [
        '#type' => 'html_tag',
        '#tag' => 'h2',
        '#value' => $this->t('Menu Preview - @menu', ['@menu' => $menu]),
      ],
      'description' => [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#value' => $this->t('This is a preview of how the menu would look alongside some content.'),
      ],
    ];

    // Fetch nodes for dropdown.
    $node_storage = \Drupal::entityTypeManager()->getStorage('node');
    $nids = $node_storage->getQuery()->accessCheck(TRUE)->execute();
    $nodes = $node_storage->loadMultiple($nids);

    // Build options: one stable "default" key + stringified node IDs.
    $options = [
      'select' => $this->t('Default'),
      'default' => $this->t('Default Content'),
    ];
    foreach ($nodes as $nid => $node_obj) {
      $options[(string) $nid] = $node_obj->label();
    }

    // Get selected node from query parameter (?preview_nid=...).
    $selected = \Drupal::request()->query->get('preview_nid', 'default');

    // Cast numerics to string to match our option keys.
    if (is_numeric($selected)) {
      $selected = (string) $selected;
    }

    // If the value isn't a valid option, fall back to 'default'.
    if (!isset($options[$selected])) {
      $selected = 'select';
    }

    // Dropdown render element.
    $dropdown = [
      '#type' => 'select',
      '#title' => $this->t('Change Preview Page'),
      '#options' => $options,
      '#default_value' => $selected,
      '#value' => $selected,
      '#attributes' => [ 'onchange' => "window.location.href = window.location.pathname + '?preview_nid=' + this.value" ],
    ];

    // Wrap the dropdown in a container for styling.
    $form = [
      '#type' => 'container',
      'dropdown' => $dropdown,
      '#attributes' => [
        'class' => ['centered-content'],
      ],
    ];

    // Load the preview node only if a node was chosen.
    if ($selected !== 'default') {
      $node = $node_storage->load((int) $selected);
      if ($node) {
        $node_view = \Drupal::entityTypeManager()->getViewBuilder('node')->view($node);
        $node_view['#cache']['max-age'] = 0;
        $sample_content['node'] = $node_view;
      }
    }

    \Drupal::messenger()->addStatus(t('You are previewing Menu Revision: @revisionid', ['@revisionid' => $menu_revision_id]));

    return [
      $form,
      $sample_content,
    ];

  }

  public function deleteDraftMenuItemRevision($menu,$menu_link_content_id)
  {
    $menu_item_link_revisions = \Drupal::service('menu_revisions.manager');
    $menu_item_link_revisions->deleteDraftMenuLinkRevision($menu,$menu_link_content_id);
    \Drupal::messenger()->addStatus(t('Menu link for this Draft is removed.'));
    return $this->redirect('entity.menu.draft', ['menu' => $menu]);
  }
}
