<?php

namespace Drupal\menu_revisions\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Url;
use Drupal\menu_revisions\Services\MenuRevisionManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for menu revision operations.
 */
class MenuRevisionController extends ControllerBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The menu revision manager.
   *
   * @var \Drupal\menu_revisions\Services\MenuRevisionManagerInterface
   */
  protected $menuRevisionManager;

  /**
   * Constructs a MenuRevisionController object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\menu_revisions\Services\MenuRevisionManagerInterface $menu_revision_manager
   *   The menu revision manager.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    DateFormatterInterface $date_formatter,
    MenuRevisionManagerInterface $menu_revision_manager
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->dateFormatter = $date_formatter;
    $this->menuRevisionManager = $menu_revision_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('date.formatter'),
      $container->get('menu_revisions.manager')
    );
  }

  /**
   * Displays a list of revisions for a specific menu.
   *
   * @param string $menu
   *   The menu machine name.
   *
   * @return array
   *   A render array.
   */
  public function revisionList($menu) {
    // Verify the menu exists
    $menu_entity = $this->entityTypeManager->getStorage('menu')->load($menu);
    if (!$menu_entity) {
      return [
        '#markup' => $this->t('Menu not found.'),
      ];
    }

    // Get all revisions for this menu
    $query = $this->entityTypeManager->getStorage('menu_revision')->getQuery()
      ->accessCheck(TRUE)
      ->condition('menu_name', $menu)
      ->sort('created', 'DESC');
    $revision_ids = $query->execute();

    $rows = [];

    // Get current default revision for this menu
    $default_revision = $this->menuRevisionManager->getDefaultRevision($menu);
    $default_revision_id = $default_revision ? $default_revision->id() : NULL;

    if (!empty($revision_ids)) {
      $revisions = $this->entityTypeManager->getStorage('menu_revision')->loadMultiple($revision_ids);
      $database = \Drupal::database();

      foreach ($revisions as $revision) {
        $row = [];

        // Revision ID and default status
        $activeRevisionId = $this->menuRevisionManager->getLatestActiveMenuRevision($menu);

        $is_default = ($revision->id() == $default_revision_id);
        $is_active = ($activeRevisionId == $revision->id());
        $status = '';

        if ($is_default) {
          $status .=  $this->t(' (Current Revision)');
        }
        if ($is_active) {
          $status .= $this->t(' (Active Revision)');
        }

        $row[] = $revision->id() . $status;
        $row[] = $this->dateFormatter->format($revision->getCreatedTime(), 'short');

        // Author
        $row[] = [
          'data' => [
            '#theme' => 'username',
            '#account' => $revision->getOwner(),
          ],
        ];

        // Label
        $row[] = $revision->label();

        // Item Count
        $item_count = $database->select('menu_revision_link', 'mrl')
          ->condition('menu_revision_id', $revision->id())
          ->countQuery()
          ->execute()
          ->fetchField();

        $row[] = $item_count;

        // Description
        $description = $revision->get('description')->value;
        $row[] = !empty($description) ? $description : $this->t('No description');

        // Operations
        $links = [];
        if (!$is_default) {
          $links['revert'] = [
            'title' => $this->t('Revert to Draft'),
            'url' => Url::fromRoute('entity.menu_revisions.revert_form', [
              'menu' => $menu,
              'menu_revision' => $revision->id(),
            ]),
          ];

          $links['view'] = [
            'title' => $this->t('View structure'),
            'url' => Url::fromRoute('entity.menu_revisions.view_structure', [
              'menu' => $menu,
              'menu_revision' => $revision->id(),
            ]),
          ];
        }

        $links['preview'] = [
          'title' => $this->t('Preview'),
          'url' => Url::fromRoute('menu_revisions.revision_preview', [
            'menu' => $menu,
            'menu_revision_id' => $revision->id(),
          ]),
        ];

        $row[] = [
          'data' => [
            '#type' => 'operations',
            '#links' => $links,
          ],
        ];

        // Apply row classes based on state and default status
        $row_classes = [];
        if ($is_default) {
          $row_classes[] = 'menu-revision-default';
        }

        // Add the row with attributes
        $rows[] = [
          'data' => $row,
          'class' => $row_classes,
        ];
      }
    }

    $build['revisions_table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Revision'),
        $this->t('Date'),
        $this->t('Author'),
        $this->t('Label'),
        $this->t('Items'),
        $this->t('Description'),
        $this->t('Operations'),
      ],
      '#rows' => $rows,
      '#empty' => $this->t('No revisions available for this menu.'),
      '#attached' => [
        'library' => ['menu_revisions/menu_revisions.admin'],
      ],
    ];

    return $build;
  }

  /**
   * Title callback for the revision list page.
   *
   * @param string $menu
   *   The menu machine name.
   *
   * @return string
   *   The page title.
   */
  public function revisionListTitle($menu) {
    $menu_entity = $this->entityTypeManager->getStorage('menu')->load($menu);
    if ($menu_entity) {
      return $this->t('Revisions for %title menu', ['%title' => $menu_entity->label()]);
    }
    return $this->t('Menu revisions');
  }

  /**
   * Displays the structure of a specific menu revision.
   *
   * @param string $menu
   *   The menu machine name.
   * @param int $menu_revision
   *   The menu revision ID.
   *
   * @return array
   *   A render array.
   */
  public function viewRevisionStructure($menu, $menu_revision) {
    $revision = $this->entityTypeManager->getStorage('menu_revision')->load($menu_revision);

    if (!$revision) {
      return [
        '#markup' => $this->t('Menu revision not found.'),
      ];
    }

    $database = \Drupal::database();

    // Get hierarchy data
    $hierarchy_data = $database->select('menu_revision_hierarchy', 'mrh')
      ->fields('mrh')
      ->condition('menu_revision_id', $menu_revision)
      ->execute()
      ->fetchAll();

    $items_by_parent = [];
    $all_items = [];

    // First pass: collect all items
    foreach ($hierarchy_data as $item) {
      $all_items[$item->plugin_id] = [
        'id' => $item->menu_link_content_id,
        'uuid' => $item->uuid,
        'parent' => $item->parent,
        'weight' => $item->weight,
        'title' => '(Unknown)', // Will populate in next step
      ];

      // Group by parent
      $parent = !empty($item->parent) ? $item->parent : '';
      if (!isset($items_by_parent[$parent])) {
        $items_by_parent[$parent] = [];
      }
      $items_by_parent[$parent][] = $item->plugin_id;
    }

    // Get titles from revision links
    $link_ids = array_column($hierarchy_data, 'menu_link_content_id');

    if (!empty($link_ids)) {
      $links_data = $database->select('menu_revision_link', 'mrl')
        ->fields('mrl')
        ->condition('menu_revision_id', $menu_revision)
        ->condition('menu_link_content_id', $link_ids, 'IN')
        ->execute();

      $menu_link_storage = $this->entityTypeManager->getStorage('menu_link_content');

      foreach ($links_data as $link) {
        $menu_link_revision = $menu_link_storage->loadRevision($link->menu_link_revision_id);
        if ($menu_link_revision) {
          $plugin_id = 'menu_link_content:' . $menu_link_revision->uuid();
          if (isset($all_items[$plugin_id])) {
            $all_items[$plugin_id]['title'] = $menu_link_revision->getTitle();
          }
        }
      }
    }

    // Build a tree representation
    $build = [
      '#theme' => 'item_list',
      '#title' => $this->t('Menu structure in revision @id', ['@id' => $menu_revision]),
      '#items' => $this->buildMenuTree('', $items_by_parent, $all_items),
      '#attributes' => ['class' => ['menu-revision-structure']],
    ];

    $build['back'] = [
      '#type' => 'link',
      '#title' => $this->t('Back to revision list'),
      '#url' => Url::fromRoute('entity.menu_revisions.collection', ['menu' => $menu]),
      '#attributes' => ['class' => ['button']],
    ];

    $build['#attached']['library'][] = 'menu_revisions/menu_revisions.admin';

    return $build;
  }

  /**
   * Recursively builds a menu tree structure.
   *
   * @param string $parent
   *   The parent plugin ID.
   * @param array $items_by_parent
   *   Items grouped by parent ID.
   * @param array $all_items
   *   All items with their data.
   *
   * @return array
   *   A nested array representing the menu tree.
   */
  protected function buildMenuTree($parent, array $items_by_parent, array $all_items) {
    $tree = [];

    if (!isset($items_by_parent[$parent])) {
      return $tree;
    }

    $items = $items_by_parent[$parent];

    // Sort by weight
    usort($items, function ($a, $b) use ($all_items) {
      return $all_items[$a]['weight'] <=> $all_items[$b]['weight'];
    });

    foreach ($items as $plugin_id) {
      $item = $all_items[$plugin_id];

      $children = $this->buildMenuTree($plugin_id, $items_by_parent, $all_items);

      if (!empty($children)) {
        $tree[] = [
          '#markup' => $item['title'] . ' (' . $item['weight'] . ')',
          '#wrapper_attributes' => ['class' => ['menu-revision-item']],
          'children' => [
            '#theme' => 'item_list',
            '#items' => $children,
            '#attributes' => ['class' => ['menu-revision-children']],
          ],
        ];
      }
      else {
        $tree[] = [
          '#markup' => $item['title'] . ' (' . $item['weight'] . ')',
          '#wrapper_attributes' => ['class' => ['menu-revision-item']],
        ];
      }
    }

    return $tree;
  }
}
