<?php

namespace Drupal\menu_revisions\Form;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Menu\MenuLinkTreeElement;
use Drupal\Core\Render\Element;
use Drupal\Core\Url;
use Drupal\Component\Utility\SortArray;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Link;
use Drupal\Core\Menu\MenuTreeParameters;
use Drupal\menu_link_content\Plugin\Menu\MenuLinkContent;
use Drupal\menu_revisions\Entity\MenuRevisionLinkTree;
use Drupal\menu_revisions\Utils\MenuTreeUtil;
use Drupal\menu_ui\MenuForm;
use Symfony\Component\DependencyInjection\ContainerInterface;



/**
 * Form for editing a menu in draft state.
 */
class MenuDraftForm extends MenuForm
{

    /**
     * The menu revision manager.
     *
     * @var \Drupal\menu_revisions\Services\MenuRevisionManagerInterface
     */
    protected $menuRevisionManager;

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        $instance = parent::create($container);
        $instance->menuRevisionManager = $container->get('menu_revisions.manager');
        return $instance;
    }

    /**
     * {@inheritdoc}
     */
    public function form(array $form, FormStateInterface $form_state)
    {
        $menu = $this->entity;
        //$form['label']['#attributes'] = ['disabled' => 'disabled'];
        //$form['description']['#attributes'] = ['disabled' => 'disabled'];
        //based on which route the form is loaded from, we will either load the draft revision or the active revision
        $route_match = \Drupal::routeMatch();
        // Get draft revision
        if ($route_match->getRouteName() === 'entity.menu.draft')
        {
            $revision = $this->getDraftRevision($menu->id());
        }
        else if ($route_match->getRouteName() === 'entity.menu.active_edit')
        {
            $revision = $this->getActiveRevision($menu->id());
            if (!$revision) {
                //revision does not exist yet, this can happen if no draft has ever been created, so create one from the current menu
                $revision_id = $this->menuRevisionManager->createRevisionFromMenu($menu->id());
                //immediately set the revision as the published revision
                $this->menuRevisionManager->publishDraftMenu($menu->id());
                $revision = $this->entityTypeManager->getStorage('menu_revision')->load($revision_id);
            }
        }
        // if neither route, do nothing
        else
        {
            return;
        }

        $form = parent::form($form, $form_state);

        // this is not really needed since form will return blank
        if (!$revision) {
            $revision_id = $this->menuRevisionManager->createRevisionFromMenu($menu->id());
            $revision = $this->entityTypeManager->getStorage('menu_revision')->load($revision_id);
        }

        // Store the revision ID in form state
        $form_state->set('menu_revision_id', $revision->id());

        if ($route_match->getRouteName() === 'entity.menu.draft')
        {
            $title = 'DRAFT MENU';
            $class = 'menu-revision-draft-indicator';
            $message = 'You are editing a draft version of this menu. Changes will not be visible to site visitors until you publish the menu.';
        }
        else if ($route_match->getRouteName() === 'entity.menu.active_edit')
        {
            $title = 'ACTIVE MENU';
            $class = 'menu-revision-published-indicator';
            $message = 'You are editing the live version of this menu. Changes will be immediately visible to site visitors.';
        }


        // Add draft status indicator
        $form['status_indicator'] = [
            '#type' => 'container',
            '#weight' => -100,
            '#attributes' => [
                'class' => ['menu-revision-status-indicator', $class],
            ],
            'message' => [
                '#markup' => $this->t('<strong>' . $title . '</strong> - ' . $message),
            ],
        ];

        // Add CSS for styling the indicator
        $form['#attached']['library'][] = 'menu_revisions/menu_revisions.admin';

        return $form;
    }


    /**
     * Builds a menu tree from revision data.
     */
    protected function buildMenuTreeFromRevision($revision_id, $menu_name, array $hierarchy_data)
    {
        $menu_tree = [];

        // Get menu links from revision
        $menu_revision_links = \Drupal::service('menu_revisions.manager')
            ->getMenuTreeLinksFromMenuRevision($menu_name, $revision_id);

        if (empty($menu_revision_links)) {
            return $menu_tree;
        }

        // Build tree structure from flat list
        $items_by_parent = [];
        foreach ($hierarchy_data as $uuid => $item) {

            $parent = !empty($item['parent']) ? $item['parent'] : '';
            if (!isset($items_by_parent[$parent])) {
                $items_by_parent[$parent] = [];
            }
            $items_by_parent[$parent][] = $uuid;
        }

        // Create tree elements recursively
        $build_tree = function ($parent = '', $depth = 1) use (&$build_tree, $items_by_parent, $menu_revision_links, $hierarchy_data) {
            $tree = [];

            if (!isset($items_by_parent[$parent])) {
                return $tree;
            }

            // Sort items by weight
            $items = $items_by_parent[$parent];
            usort($items, function ($a, $b) use ($hierarchy_data) {
                return $hierarchy_data[$a]['weight'] <=> $hierarchy_data[$b]['weight'];
            });

            foreach ($items as $uuid) {
                if (!isset($menu_revision_links[$uuid])) {
                    continue;
                }

                $link = $menu_revision_links[$uuid];
                $plugin_id = $link->getPluginId();

                // Create menu tree element
                $tree_element = new MenuLinkTreeElement($link, TRUE, $depth, TRUE, []);

                // Add children recursively
                $children = $build_tree($plugin_id, $depth + 1);
                if (!empty($children)) {
                    $tree_element->subtree = $children;
                    $tree_element->hasChildren = TRUE;
                }

                $tree[$plugin_id] = $tree_element;
            }

            return $tree;
        };

        // Build the tree starting from top level items (empty parent)
        return $build_tree();
    }



    /**
     * {@inheritdoc}
     */
    protected function actions(array $form, FormStateInterface $form_state)
    {
        $actions = parent::actions($form, $form_state);

        if (isset($actions['submit'])) {
            $actions['submit']['#value'] = $this->t('Save Draft');
            // Use our draft save handler - disabled, changed logical workfow
            $actions['submit']['#submit'] = ['::saveDraftSubmit'];
        }

        // Change the default "Save" button text.
        $route_match = \Drupal::routeMatch();
        if ($route_match->getRouteName() === 'entity.menu.active_edit') {
            unset($actions['submit']);
        }
        else
        {
            // Add a new "Save and Publish" button - ONLY for publishers
            $actions['publish'] = [
                '#type' => 'submit',
                '#value' => $this->t('Save and Publish'),
                '#button_type' => 'primary',
                '#weight' => 5,
                '#submit' => ['::publishSubmit'],
                '#access' => $this->currentUser()->hasPermission('administer menu revisions'),
            ];

            $menuManager = \Drupal::service('menu_revisions.manager');
            $revisionID = $menuManager->getDefaultRevision($this->entity->id())->id();
            if ($revisionID)
            {
                // Preview available to anyone who can edit drafts OR administer menu revisions
                $can_preview = $this->currentUser()->hasPermission('edit draft menus') 
                    || $this->currentUser()->hasPermission('administer menu revisions');
                
                $actions['preview'] = [
                    '#type' => 'link',
                    '#title' => $this->t('Preview'),
                    '#attributes' => ['class' => ['button'],'target' => '_blank'],
                    '#weight' => 5,
                    '#url' => Url::fromRoute('menu_revisions.revision_preview', ['menu' => $this->entity->id(), 'menu_revision_id' => $revisionID]),
                    '#access' => $can_preview,
                ];
            }


        }



        return $actions;
    }

    /**
    /**
     * {@inheritdoc}
     */
    public function saveDraftData(array &$form, FormStateInterface $form_state, $showStatus=TRUE) {
        // First save the menu normally.
        parent::submitForm($form, $form_state);
        $menu_name = $this->entity->id();
        $revision_id = null;

        try {
            // Create a new revision for this menu.
            $revision_id = $this->menuRevisionManager->createRevisionFromMenu($menu_name);
        }
        catch (\Exception $e) {
            $this->messenger()->addError($this->t('Failed to create menu draft: @message', ['@message' => $e->getMessage()]));
            return $revision_id;
        }
        if (!$revision_id) {
            $this->messenger()->addError($this->t('Failed to create menu draft revision.'));
            return $revision_id;
        }

        return $revision_id;


    }



    /**
     * Submit handler for "Save draft".
     */
    public function saveDraftSubmit(array &$form, FormStateInterface $form_state)
    {
        //save menu title and description changes
        $menu = $this->entity;
        //validate the submission, prbly should use the validate hook but since we are overwritting the whole funciton it's not needed.
        $valid = $this->validateMenuForUpdate($menu->links);
        if (!$valid) {
          return;
        }
        $menu->save();
        $this->saveDraftData($form, $form_state);
    }


    /**
     * Submit handler for the Publish button.
     */
    public function publishSubmit(array &$form, FormStateInterface $form_state)
    {

        $menu_name = $this->entity->id();

        // First process the draft form submission
        $new_revision_id = $this->saveDraftData($form, $form_state,false);

        try {
            // Create a published revision
            // $new_revision_id = $this->menuRevisionManager->createRevisionFromMenu($menu_name, TRUE);

            // Mark the revision as published (status = 1)
            $new_rev = $this->entityTypeManager->getStorage('menu_revision')->load($new_revision_id);
            if ($new_rev) {
                $new_rev->set('status', 1);
                $new_rev->save();

                // Apply the revision to make it live
                $this->menuRevisionManager->revertMenuToRevision($new_revision_id);
                $this->messenger()->addStatus($this->t('Draft (revision @rid) has been published.', ['@rid' => $new_revision_id]));
            } else {
                $this->messenger()->addError($this->t('Failed to create published revision.'));
            }
        } catch (\Exception $e) {
            $this->messenger()->addError($this->t('Failed to publish menu: @msg', ['@msg' => $e->getMessage()]));
        }

        $form_state->setRedirect('entity.menu.edit_form', ['menu' => $menu_name]);
    }

    /**
     * Gets the latest draft revision for a menu.
     */
    protected function getLatestDraftRevision($menu_name)
    {
        $menuManager = \Drupal::service('menu_revisions.manager');
        $draftRevision = $menuManager->getDefaultRevision($menu_name);
        //create new draft if it does not exist.
        if (!$draftRevision) {
            $menu_revision_id = $menuManager->createRevisionFromMenu($menu_name);
            $draftRevision = $menuManager->getDefaultRevision($menu_name);
        }

        return $draftRevision;
    }

    /**
     * Get current active revision (published)
     *
     * @param $menu_name
     * @return \Drupal\Core\Entity\EntityInterface|null
     * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
     * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
     */
    protected function getActiveRevision($menu_name) {
        $menuManager = \Drupal::service('menu_revisions.manager');
        $revisionID = $menuManager->getLatestActiveMenuRevision($menu_name);
        return $this->entityTypeManager->getStorage('menu_revision')->load($revisionID);
    }

    protected function getDraftRevision($menu_name) {
        // Your existing code to get a draft revision
        $menuManager = \Drupal::service('menu_revisions.manager');
        $draftRevision = $this->getLatestDraftRevision($menu_name);

        if ($draftRevision) {
            $ids = [$draftRevision->id()];
        }

        $revisionID = reset($ids);
        return $this->entityTypeManager->getStorage('menu_revision')->load($revisionID);
    }



    /**
     * Form constructor to edit an entire menu tree at once.
     *
     * Shows for one menu the menu links accessible to the current user and
     * relevant operations.
     *
     * This form constructor can be integrated as a section into another form. It
     * relies on the following keys in $form_state:
     * - menu: A menu entity.
     * - menu_overview_form_parents: An array containing the parent keys to this
     *   form.
     * Forms integrating this section should call menu_overview_form_submit() from
     * their form submit handler.
     */
    protected function buildOverviewForm(array &$form, FormStateInterface $form_state) {
        // Ensure that menu_overview_form_submit() knows the parents of this form
        // section.
        if (!$form_state->has('menu_overview_form_parents')) {
            $form_state->set('menu_overview_form_parents', []);
        }

        $form['#attached']['library'][] = 'menu_ui/drupal.menu_ui.adminforms';

        $route_match = \Drupal::routeMatch();
        if ($route_match->getRouteName() === 'entity.menu.draft')
        {
            $tree = $this->menuTree->load($this->entity->id(), new MenuTreeParameters());
        }
        else if ($route_match->getRouteName() === 'entity.menu.active_edit')
        {
            /*
            $menuManager = \Drupal::service('menu_revisions.manager');
            $revisionID = $menuManager->getLatestActiveMenuRevision($this->entity->id());

            $menuItemsContent = $menuManager->generateMenuFromRevision($this->entity->id(), $revisionID);
            $menutreeutil = new MenuTreeUtil();
            $tree = $menutreeutil->generateMenuTree( $menu_name = $this->entity->id(), $revisionID,$menuItemsContent);
            */
            $tree = [];
        }
        else
        {
            // I know it's repeat first condition, but this is to setup for future behavior in a nice reverse-psych way
            $tree = $this->menuTree->load($this->entity->id(), new MenuTreeParameters());
        }

        // We indicate that a menu administrator is running the menu access check.
        $this->getRequest()->attributes->set('_menu_admin', TRUE);
        $manipulators = [
            // Use a dedicated menu tree access check manipulator as users editing
            // this form, granted with 'administer menu' permission, should be able to
            // access menu links with inaccessible routes. The default menu tree
            // manipulator only allows the access to menu links with accessible routes.
            // @see \Drupal\Core\Menu\DefaultMenuLinkTreeManipulators::checkAccess()
            // @see \Drupal\menu_ui\Menu\MenuUiMenuTreeManipulators::checkAccess()
            ['callable' => 'menu_ui.menu_tree_manipulators:checkAccess'],
            ['callable' => 'menu.default_tree_manipulators:generateIndexAndSort'],
        ];
        $tree = $this->menuTree->transform($tree, $manipulators);
        $this->getRequest()->attributes->set('_menu_admin', FALSE);

        // Determine the delta; the number of weights to be made available.
        $count = function (array $tree) {
            $sum = function ($carry, MenuLinkTreeElement $item) {
                return $carry + $item->count();
            };
            return array_reduce($tree, $sum);
        };
        $delta = max($count($tree), 50);

        $form['links'] = [
            '#type' => 'table',
            '#theme' => 'table__menu_overview',
            '#header' => [
                $this->t('Menu link'),
                [
                    'data' => $this->t('Enabled'),
                    'class' => ['checkbox'],
                ],
                $this->t('Weight'),
                [
                    'data' => $this->t('Operations'),
                    'colspan' => 3,
                ],
            ],
            '#attributes' => [
                'id' => 'menu-overview',
            ],
            '#tabledrag' => [
                [
                    'action' => 'match',
                    'relationship' => 'parent',
                    'group' => 'menu-parent',
                    'subgroup' => 'menu-parent',
                    'source' => 'menu-id',
                    'hidden' => TRUE,
                    'limit' => $this->menuTree->maxDepth() - 1,
                ],
                [
                    'action' => 'order',
                    'relationship' => 'sibling',
                    'group' => 'menu-weight',
                ],
            ],
        ];



        // Get draft revision
        if ($route_match->getRouteName() === 'entity.menu.draft') {
            $form['links']['#empty'] = $this->t('There are no menu links yet. <a href=":url">Add link</a>.', [
                ':url' => Url::fromRoute('entity.menu.add_link_form', ['menu' => $this->entity->id()], [
                    'query' => ['destination' => $this->entity->toUrl('edit-form')->toString()],
                ])->toString(),
            ]);
        }
        $links = $this->buildOverviewTreeForm($tree, $delta);

        // Get the menu links which have pending revisions, and disable the
        // tabledrag if there are any.
        $edited_ids = array_filter(array_map(function ($element) {
            return is_array($element) && isset($element['#item']) && $element['#item']->link instanceof MenuLinkContent ? $element['#item']->link->getMetaData()['entity_id'] : NULL;
        }, $links));
        $pending_menu_link_ids = array_intersect($this->menuLinkContentStorage->getMenuLinkIdsWithPendingRevisions(), $edited_ids);
        if ($pending_menu_link_ids) {
            $form['help'] = [
                '#type' => 'container',
                'message' => [
                    '#markup' => $this->formatPlural(
                        count($pending_menu_link_ids),
                        '%capital_name contains 1 menu link with pending revisions. Manipulation of a menu tree having links with pending revisions is not supported, but you can re-enable manipulation by getting each menu link to a published state.',
                        '%capital_name contains @count menu links with pending revisions. Manipulation of a menu tree having links with pending revisions is not supported, but you can re-enable manipulation by getting each menu link to a published state.',
                        [
                            '%capital_name' => $this->entity->label(),
                        ]
                    ),
                ],
                '#attributes' => ['class' => ['messages', 'messages--warning']],
                '#weight' => -10,
            ];

            unset($form['links']['#tabledrag']);
            unset($form['links']['#header'][2]);
        }

        foreach (Element::children($links) as $id) {
            if (isset($links[$id]['#item'])) {
                $element = $links[$id];

                $is_pending_menu_link = isset($element['#item']->link->getMetaData()['entity_id'])
                    && in_array($element['#item']->link->getMetaData()['entity_id'], $pending_menu_link_ids);

                $form['links'][$id]['#item'] = $element['#item'];

                // TableDrag: Mark the table row as draggable.
                $form['links'][$id]['#attributes'] = $element['#attributes'];
                $form['links'][$id]['#attributes']['class'][] = 'draggable';

                if ($is_pending_menu_link) {
                    $form['links'][$id]['#attributes']['class'][] = 'color-warning';
                    $form['links'][$id]['#attributes']['class'][] = 'menu-link-content--pending-revision';
                }

                // TableDrag: Sort the table row according to its existing/configured weight.
                $form['links'][$id]['#weight'] = $element['#item']->link->getWeight();

                // Add special classes to be used for tabledrag.js.
                $element['parent']['#attributes']['class'] = ['menu-parent'];
                $element['weight']['#attributes']['class'] = ['menu-weight'];
                $element['id']['#attributes']['class'] = ['menu-id'];

                $form['links'][$id]['title'] = [
                    [
                        '#theme' => 'indentation',
                        '#size' => $element['#item']->depth - 1,
                    ],
                    $element['title'],
                ];

                $form['links'][$id]['enabled'] = $element['enabled'];
                $form['links'][$id]['enabled']['#wrapper_attributes']['class'] = ['checkbox', 'menu-enabled'];

                // Disallow changing the publishing status of a pending revision.
                if ($is_pending_menu_link) {
                    $form['links'][$id]['enabled']['#access'] = FALSE;
                }

                // Disallow changing the publishing status of a pending revision or if the link has children.
                if (!empty($element['#item']->subtree)) {
                    $form['links'][$id]['enabled'] = [
                        '#type' => 'checkbox',
                        '#title_display' => 'invisible',
                        '#attributes' => ['disabled' => 'disabled',],
                        '#value' => TRUE,
                        '#input' => FALSE,
                    ];
                    $form['links'][$id]['enabled']['#wrapper_attributes']['class'] = ['checkbox', 'menu-enabled'];
                }

                if (!$pending_menu_link_ids) {
                    $form['links'][$id]['weight'] = $element['weight'];
                }

                // Operations (dropbutton) column.
                $form['links'][$id]['operations'] = $element['operations'];

                $form['links'][$id]['id'] = $element['id'];
                $form['links'][$id]['parent'] = $element['parent'];
            }
        }
        return $form;
    }

    /**
     * Recursive helper function for buildOverviewForm().
     *
     * @param \Drupal\Core\Menu\MenuLinkTreeElement[] $tree
     *   The tree retrieved by \Drupal\Core\Menu\MenuLinkTreeInterface::load().
     * @param int $delta
     *   The default number of menu items used in the menu weight selector is 50.
     *
     * @return array
     *   The overview tree form.
     */
    protected function buildOverviewTreeForm($tree, $delta) {
        $form = &$this->overviewTreeForm;
        $tree_access_cacheability = new CacheableMetadata();

        foreach ($tree as $element) {
            $tree_access_cacheability = $tree_access_cacheability->merge(CacheableMetadata::createFromObject($element->access));

            // Only render accessible links.
            if (!$element->access->isAllowed()) {
                continue;
            }

            /** @var \Drupal\Core\Menu\MenuLinkInterface $link */
            $link = $element->link;
            if ($link) {
                $id = 'menu_plugin_id:' . $link->getPluginId();

                $form[$id]['#item'] = $element;
                $form[$id]['#attributes'] = $link->isEnabled() ? ['class' => ['menu-enabled']] : ['class' => ['menu-disabled']];
                $form[$id]['title'] = Link::fromTextAndUrl($link->getTitle(), $link->getUrlObject())->toRenderable();
                if (!$link->isEnabled()) {
                    $form[$id]['title']['#suffix'] = ' (' . $this->t('disabled') . ')';
                }
                // @todo Remove this in https://www.drupal.org/node/2568785.
                elseif ($id === 'menu_plugin_id:user.logout') {
                    $form[$id]['title']['#suffix'] = ' (' . $this->t('<q>Log in</q> for anonymous users') . ')';
                }
                // @todo Remove this in https://www.drupal.org/node/2568785.
                elseif (($url = $link->getUrlObject()) && $url->isRouted() && $url->getRouteName() == 'user.page') {
                    $form[$id]['title']['#suffix'] = ' (' . $this->t('logged in users only') . ')';
                }

                $form[$id]['enabled'] = [
                    '#type' => 'checkbox',
                    '#title' => $this->t('Enable @title menu link', ['@title' => $link->getTitle()]),
                    '#title_display' => 'invisible',
                    '#default_value' => $link->isEnabled(),
                ];
                $form[$id]['weight'] = [
                    '#type' => 'weight',
                    '#delta' => $delta,
                    '#default_value' => $link->getWeight(),
                    '#title' => $this->t('Weight for @title', ['@title' => $link->getTitle()]),
                    '#title_display' => 'invisible',
                ];
                $form[$id]['id'] = [
                    '#type' => 'hidden',
                    '#value' => $link->getPluginId(),
                ];
                $form[$id]['parent'] = [
                    '#type' => 'hidden',
                    '#default_value' => $link->getParent(),
                ];

                $operations = $link->getOperations();
                // remove delete operation from this form... yes it is a bit hacky but it works
                unset($operations['delete']);

                if ($element->depth < $this->menuTree->maxDepth()) {

                    $add_link_url = Url::fromRoute(
                        'entity.menu.add_link_form',
                        ['menu' => $this->entity->id()],
                        ['query' => ['parent' => $link->getPluginId()]]
                    );
                    $operations += [
                        'add-child' => [
                            'title' => $this->t('Add child'),
                            'weight' => 20,
                            'url' => $add_link_url,
                        ],
                    ];

                    uasort($operations, [SortArray::class, 'sortByWeightElement']);
                }
                foreach ($operations as $key => $operation) {
                    if (!isset($operations[$key]['query'])) {
                        // Bring the user back to the menu overview.
                        $operations[$key]['query'] = $this->getDestinationArray();
                    }
                }

                $route_match = \Drupal::routeMatch();
                // Get draft revision
                if ($route_match->getRouteName() === 'entity.menu.draft') {
                    $form[$id]['operations'] = [
                        '#type' => 'operations',
                        '#links' => $operations,
                    ];
                }
            }

            if ($element->subtree) {
                $this->buildOverviewTreeForm($element->subtree, $delta);
            }
        }

        $tree_access_cacheability
            ->merge(CacheableMetadata::createFromRenderArray($form))
            ->applyTo($form);


        return $form;
    }

    /**
     * Helper: returns the latest published menu_revision id for a menu, or NULL.
     */
    protected function getLatestPublishedRevisionId($menu_name)
    {
        $query = $this->entityTypeManager->getStorage('menu_revision')->getQuery()
            ->accessCheck(TRUE)
            ->condition('menu_name', $menu_name)
            ->condition('status', 1)
            ->sort('created', 'DESC')
            ->range(0, 1);
        $ids = $query->execute();
        if (empty($ids)) {
            return NULL;
        }
        return (int)reset($ids);
    }

  /**
   * Validate the tree structure to make sure it's valid before saving.
   * @param array $menuItemLinks
   * @return true
   */
    protected function validateMenuForUpdate(array $menuItemLinks)
    {
      $valid = true;
      foreach ($menuItemLinks as $menuItemLink) {

        foreach ($menuItemLinks as $potentialChild) {
          if (isset($potentialChild['parent']) && $potentialChild['parent'] === $menuItemLink['id']) {

            if (array_key_exists('enabled',$menuItemLink) && $menuItemLink['enabled'] == 0)
            {
              $valid = false;
            }

            // $menuItemLink has at least one child
            // You can add your logic here, e.g., set a flag or process accordingly
          }
        }

      }

      if (!$valid) {
        $this->messenger()->addError($this->t('Menu structure is invalid. A disabled menu item cannot have enabled children. Please correct the menu structure and try again.'));
      }

      return $valid;
    }
}
