<?php

namespace Drupal\menu_revisions\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\menu_revisions\Services\MenuRevisionManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Datetime\DateFormatterInterface;

/**
 * Provides a form for reverting a menu to a previous revision.
 */
class MenuRevisionRevertForm extends ConfirmFormBase {

  /**
   * The menu revision manager.
   *
   * @var \Drupal\menu_revisions\Services\MenuRevisionManagerInterface
   */
  protected $menuRevisionManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The menu machine name.
   *
   * @var string
   */
  protected $menuName;

  /**
   * The menu revision entity.
   *
   * @var \Drupal\menu_revisions\Entity\MenuRevision
   */
  protected $menuRevision;

  /**
   * Constructs a MenuRevisionRevertForm object.
   *
   * @param \Drupal\menu_revisions\Services\MenuRevisionManagerInterface $menu_revision_manager
   *   The menu revision manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   */
  public function __construct(
    MenuRevisionManagerInterface $menu_revision_manager,
    EntityTypeManagerInterface $entity_type_manager,
    DateFormatterInterface $date_formatter
  ) {
    $this->menuRevisionManager = $menu_revision_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('menu_revisions.manager'),
      $container->get('entity_type.manager'),
      $container->get('date.formatter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'menu_revisions_revert_confirm';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $menu = NULL, $menu_revision = NULL) {
    $this->menuName = $menu;

    // Load the menu revision entity.
    $this->menuRevision = $this->entityTypeManager->getStorage('menu_revision')->load($menu_revision);

    if (!$this->menuRevision) {
      $this->messenger()->addError($this->t('Menu revision not found.'));
      return $this->redirect('entity.menu_revisions.collection', ['menu' => $menu]);
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    $date = $this->dateFormatter->format($this->menuRevision->getCreatedTime(), 'short');
    $menu_entity = $this->entityTypeManager->getStorage('menu')->load($this->menuName);

    return $this->t('Are you sure you want to revert the %menu menu to the revision from %date?', [
      '%menu' => $menu_entity ? $menu_entity->label() : $this->menuName,
      '%date' => $date,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('entity.menu_revisions.collection', ['menu' => $this->menuName]);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Revert');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('This will create a new revision of the menu reflecting the state from the selected revision. The current menu structure will be preserved as a revision for historical reference.');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $result = $this->menuRevisionManager->revertMenuToRevision($this->menuRevision->id());

    if ($result) {
      $this->messenger()->addStatus($this->t('The menu has been reverted to the selected revision.'));
    }
    else {
      $this->messenger()->addError($this->t('Failed to revert the menu.'));
    }

    $form_state->setRedirect('entity.menu_revisions.collection', ['menu' => $this->menuName]);
  }

}
