<?php

namespace Drupal\menu_revisions\Manipulator;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\menu_revisions\Services\MenuRevisionManagerInterface;

/**
 * Menu link tree manipulator for draft revisions.
 */
class MenuLinkDraftManipulator {

  /**
   * The menu revision manager.
   *
   * @var \Drupal\menu_revisions\Services\MenuRevisionManagerInterface
   */
  protected $menuRevisionManager;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a MenuLinkDraftManipulator object.
   *
   * @param \Drupal\menu_revisions\Services\MenuRevisionManagerInterface $menu_revision_manager
   *   The menu revision manager service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   */
  public function __construct(MenuRevisionManagerInterface $menu_revision_manager, RouteMatchInterface $route_match, ConfigFactoryInterface $config_factory) {
    $this->menuRevisionManager = $menu_revision_manager;
    $this->routeMatch = $route_match;
    $this->configFactory = $config_factory;
  }

  /**
   * Removes items that don't exist in the menu_revision_link table.
   *
   * Items that have been removed in draft mode are filtered out from the tree.
   * When menu revisions is enabled for a menu, disabled items are filtered out
   * to maintain consistent UX (no "disable" feature in menu revisions).
   *
   * @param array $tree
   *   The menu tree to filter.
   *
   * @return array
   *   The filtered menu tree.
   */
  public function removeDeleteItemFromDraft(array $tree): array {
    $menuID = $this->routeMatch->getParameter('menu');

    // If no menu parameter is available, return tree as-is.
    if (!$menuID) {
      return $tree;
    }

    $menu_id = is_string($menuID) ? $menuID : $menuID->id();
    
    // Check if menu revisions is enabled for this menu.
    if (!$this->isMenuRevisionEnabled($menu_id)) {
      // Menu revisions NOT enabled - return tree as-is (standard Drupal behavior).
      // Disabled items will show with unchecked enabled box.
      return $tree;
    }

    // Menu revisions IS enabled - apply filtering logic.
    $draft_revision = $this->menuRevisionManager->getDefaultRevision($menu_id);

    // If no draft revision exists, filter disabled items to maintain consistent UX.
    // Menu revisions doesn't have a "disable" feature - items should not appear if disabled.
    if (!$draft_revision) {
      return $this->filterDisabledItems($tree);
    }

    $revision_items = $this->menuRevisionManager->generateMenuFromRevision($menu_id, $draft_revision->id());

    // Loop through the menu tree and remove items not in the revision.
    foreach ($tree as $key => $element) {
      $plugin_id_parts = explode(':', $element->link->getPluginId());
      if (count($plugin_id_parts) < 2) {
        continue;
      }

      $uuid = $plugin_id_parts[1];
      if (!isset($revision_items[$uuid])) {
        unset($tree[$key]);
      }
      else {
        // Recursively check children.
        if (!empty($element->subtree)) {
          $element->subtree = $this->removeDeleteItemFromDraft($element->subtree);
        }
      }
    }

    return $tree;
  }

  /**
   * Checks if menu revisions is enabled for a specific menu.
   *
   * @param string $menu_id
   *   The menu ID.
   *
   * @return bool
   *   TRUE if menu revisions is enabled for this menu, FALSE otherwise.
   */
  protected function isMenuRevisionEnabled(string $menu_id): bool {
    $config = $this->configFactory->get('menu_revisions.settings');
    $selected_menus = $config->get('selected_menu');
    return is_array($selected_menus) && in_array($menu_id, $selected_menus, TRUE);
  }

  /**
   * Filters out disabled items from the menu tree.
   *
   * @param array $tree
   *   The menu tree to filter.
   *
   * @return array
   *   The filtered menu tree with disabled items removed.
   */
  protected function filterDisabledItems(array $tree): array {
    foreach ($tree as $key => $element) {
      // Remove disabled items.
      if (!$element->link->isEnabled()) {
        unset($tree[$key]);
        continue;
      }

      // Recursively filter children.
      if (!empty($element->subtree)) {
        $element->subtree = $this->filterDisabledItems($element->subtree);
      }
    }

    return $tree;
  }

}
