<?php

namespace Drupal\menu_revisions;

use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Access\AccessResultAllowed;
use Drupal\Core\Access\AccessResultForbidden;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Access\AccessResultNeutral;
use Drupal\Core\Access\AccessResultReasonInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Access\AccessResult;

/**
 * Access controller for the Menu Revision entity.
 */
class MenuRevisionAccessControlHandler extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account): AccessResultReasonInterface|AccessResultForbidden|AccessResultNeutral|AccessResult|AccessResultAllowed|AccessResultInterface {
    // Admin permission always grants access.
    if ($account->hasPermission('administer menu revisions')) {
      return AccessResult::allowed()->cachePerPermissions();
    }

    switch ($operation) {
      case 'view':
        return AccessResult::allowedIfHasPermission($account, 'view menu revisions');

      case 'update':
        // Allow update if user can edit drafts OR administer revisions
        if ($account->hasPermission('edit draft menus') || $account->hasPermission('administer menu revisions')) {
          return AccessResult::allowed()->cachePerPermissions();
        }
        return AccessResult::neutral();

      case 'delete':
        // Don't allow deleting the current default revision.
        if ($entity->isDefault()) {
          return AccessResult::forbidden('Cannot delete the current default revision');
        }
        return AccessResult::allowedIfHasPermission($account, 'delete menu revisions');
    }

    return AccessResult::neutral();
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL) {
    return AccessResult::allowedIfHasPermission($account, 'create menu revisions');
  }

}
