<?php

namespace Drupal\menu_revisions\Services;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for managing menu hierarchies during revision operations.
 */
class MenuHierarchyManager {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a new MenuHierarchyManager.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    Connection $database,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
    $this->logger = $logger_factory->get('menu_revisions');
  }

  /**
   * Captures the current hierarchy of a menu.
   *
   * @param string $menu_name
   *   The machine name of the menu.
   * @param int $menu_revision_id
   *   The ID of the menu revision entity.
   */
  public function captureMenuHierarchy($menu_name, $menu_revision_id) {
    // Get all menu links for this menu
    $menu_link_storage = $this->entityTypeManager->getStorage('menu_link_content');
    $query = $menu_link_storage->getQuery()
      ->accessCheck(TRUE)
      ->condition('menu_name', $menu_name)
      ->condition('enabled', TRUE);
    $link_ids = $query->execute();

    if (empty($link_ids)) {
      return;
    }

    $menu_links = $menu_link_storage->loadMultiple($link_ids);

    // Build a map of UUIDs to plugin IDs for lookups
    $uuid_map = [];
    foreach ($menu_links as $link) {
      $uuid = $link->uuid();
      $plugin_id = 'menu_link_content:' . $uuid;
      $uuid_map[$uuid] = $plugin_id;

      // Store hierarchy information
      $this->database->insert('menu_revision_hierarchy')
        ->fields([
          'menu_revision_id' => $menu_revision_id,
          'menu_link_content_id' => $link->id(),
          'uuid' => $uuid,
          'parent' => $link->getParentId(),
          'weight' => $link->getWeight(),
          'plugin_id' => $plugin_id,
        ])
        ->execute();
    }
  }

  /**
   * Retrieves hierarchy information for a specific menu revision.
   *
   * @param int $menu_revision_id
   *   The menu revision ID.
   *
   * @return array
   *   An array of menu item hierarchy data.
   */
  public function getMenuHierarchy($menu_revision_id) {
    $hierarchy = [];

    $result = $this->database->select('menu_revision_hierarchy', 'mrh')
      ->fields('mrh')
      ->condition('menu_revision_id', $menu_revision_id)
      ->execute();

    foreach ($result as $record) {
      $hierarchy[$record->uuid] = [
        'id' => $record->menu_link_content_id,
        'parent' => $record->parent,
        'weight' => $record->weight,
        'plugin_id' => $record->plugin_id,
      ];
    }

    return $hierarchy;
  }
}
