<?php

namespace Drupal\menu_revisions\Services;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Menu\MenuTreeParameters;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Service for managing menu revisions.
 */
class MenuRevisionManager implements MenuRevisionManagerInterface {
  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The menu hierarchy manager.
   *
   * @var \Drupal\menu_revisions\Services\MenuHierarchyManager
   */
  protected $hierarchyManager;

  /**
   * Constructs a new MenuRevisionManager.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\menu_revisions\Services\MenuHierarchyManager $hierarchy_manager
   *   The menu hierarchy manager.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    AccountProxyInterface $current_user,
    Connection $database,
    LoggerChannelFactoryInterface $logger_factory,
    MenuHierarchyManager $hierarchy_manager
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
    $this->database = $database;
    $this->logger = $logger_factory->get('menu_revision');
    $this->hierarchyManager = $hierarchy_manager;
  }

  /**
   * {@inheritdoc}
   */
  public function createRevisionFromMenu($menu_name, $status = 0): ?int {
    // Start a database transaction
    $transaction = $this->database->startTransaction('menu_revision_create');

    try {
      // 1. Create a new menu revision entity.
      $storage = $this->entityTypeManager->getStorage('menu_revision');

      // Set any existing revisions as not default.
      $existing_default = $this->getDefaultRevision($menu_name);
      if ($existing_default) {
        $existing_default->setDefault(FALSE);
        $storage->save($existing_default);
      }

      // Create the new revision entity.
      $menu_revision = $storage->create([
        'menu_name' => $menu_name,
        'label' => $this->t('Revision from @date', ['@date' => date('Y-m-d H:i:s')]),
        'uid' => $this->currentUser->id(),
        'is_default' => TRUE,
        'status' => $status,
      ]);

      $storage->save($menu_revision);

      // 2. Now capture all current menu links for this menu.
      $this->captureMenuLinkRevisions($menu_name, $menu_revision->id());

      // 3. Capture hierarchy information
      $this->hierarchyManager->captureMenuHierarchy($menu_name, $menu_revision->id());

      return $menu_revision->id();
    }
    catch (\Exception $e) {
      // Something went wrong, roll back the transaction
      $transaction->rollBack();
      $this->logger->error('Failed to create menu revision: @message', ['@message' => $e->getMessage()]);
      throw $e; // Re-throw the exception so the caller knows something failed
    }
  }

  /**
   * {@inheritdoc}
   */
  public function captureMenuLinkRevisions($menu_name, $menu_revision_id): void {
    // Get all menu link content entities for this menu.
    $menu_link_storage = $this->entityTypeManager->getStorage('menu_link_content');
    $query = $menu_link_storage->getQuery()
      ->accessCheck(TRUE)
      ->condition('menu_name', $menu_name)
      ->condition('enabled', TRUE); // Only capture enabled menu items
    $ids = $query->execute();

    if (!$ids) {
      return;
    }

    $menu_link_entities = $menu_link_storage->loadMultiple($ids);

    // Save each link in our relationship table.
    foreach ($menu_link_entities as $menu_link) {
      // Store reference to the revision.
      $this->database->insert('menu_revision_link')
        ->fields([
          'menu_revision_id' => $menu_revision_id,
          'menu_link_content_id' => $menu_link->id(),
          'menu_link_revision_id' => $menu_link->getRevisionId(),
        ])
        ->execute();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultRevision($menu_name) {

    // first check if default revision exists
    $query = $this->entityTypeManager->getStorage('menu_revision')->getQuery()
          ->accessCheck(TRUE)
          ->condition('menu_name', $menu_name)
          ->condition('is_default', TRUE);
    $ids = $query->execute();
    // if not exist, create new revision and set the latest revision as default
    if (!$ids) {
      return null;
    }

    return $this->entityTypeManager->getStorage('menu_revision')->load(reset($ids));
  }

  /**
   * {@inheritdoc}
   */
  public function revertMenuToRevision($menu_revision_id) {
    // Start a database transaction
    $transaction = $this->database->startTransaction('menu_revision_revert');

    try {
      $menu_revision = $this->entityTypeManager->getStorage('menu_revision')->load($menu_revision_id);
      if (!$menu_revision) {
        $this->logger->error('Menu revision @id not found', ['@id' => $menu_revision_id]);
        return FALSE;
      }

      $menu_name = $menu_revision->getMenuName();
      $this->logger->info('Starting reversion of menu @menu to revision @id', [
        '@menu' => $menu_name,
        '@id' => $menu_revision_id
      ]);

      // Get all menu links for this revision
      $revision_links = $this->database->select('menu_revision_link', 'mrl')
        ->fields('mrl')
        ->condition('menu_revision_id', $menu_revision_id)
        ->execute()
        ->fetchAllAssoc('menu_link_content_id');

      // Get hierarchy information for this revision
      $hierarchy_data = $this->hierarchyManager->getMenuHierarchy($menu_revision_id);

      if (empty($revision_links) || empty($hierarchy_data)) {
        $this->logger->warning('Insufficient data for revision @id', ['@id' => $menu_revision_id]);
        return FALSE;
      }

      $menu_link_storage = $this->entityTypeManager->getStorage('menu_link_content');

      // 1. Get ALL current menu links for this menu
      $query = $menu_link_storage->getQuery()
        ->accessCheck(TRUE)
        ->condition('menu_name', $menu_name);
      $all_current_ids = $query->execute();
      $all_current_links = !empty($all_current_ids) ? $menu_link_storage->loadMultiple($all_current_ids) : [];

      // Build maps for current links by UUID and ID
      $current_links_by_uuid = [];
      foreach ($all_current_links as $id => $link) {
        $current_links_by_uuid[$link->uuid()] = $link;
      }

      // Build a list of UUIDs in the revision
      $revision_uuids = array_keys($hierarchy_data);

      // First pass: Create or update menu items without setting parent relationships
      $uuid_to_entity_map = [];

      foreach ($hierarchy_data as $uuid => $item_data) {
        $link_id = $item_data['id'];

        if (isset($revision_links[$link_id])) {
          $revision_link = $revision_links[$link_id];
          $menu_link_revision = $menu_link_storage->loadRevision($revision_link->menu_link_revision_id);

          if (!$menu_link_revision) {
            $this->logger->warning('Could not load revision @rev_id for link @id', [
              '@rev_id' => $revision_link->menu_link_revision_id,
              '@id' => $link_id
            ]);
            continue;
          }

          try {
            // Check if we have an entity with this UUID already
            $existing_entity = isset($current_links_by_uuid[$uuid]) ? $current_links_by_uuid[$uuid] : NULL;

            if ($existing_entity) {
              // Update existing entity
              $this->logger->info('Updating existing menu link @id with UUID @uuid', [
                '@id' => $existing_entity->id(),
                '@uuid' => $uuid
              ]);

              foreach ($menu_link_revision->getFields(FALSE) as $field_name => $field) {
                // Skip fields we don't want to copy
                if (!in_array($field_name, ['revision_id', 'revision_created', 'revision_user', 'revision_log', 'changed', 'id', 'uuid', 'parent'])) {
                  $existing_entity->set($field_name, $field->getValue());
                }
              }

              $existing_entity->set('weight', $item_data['weight']);
              $existing_entity->set('enabled', TRUE);

              // Temporarily clear parent to avoid hierarchy issues
              $existing_entity->set('parent', '');

              $existing_entity->setNewRevision(TRUE);
              $existing_entity->revision_log = 'Reverted as part of menu revision ' . $menu_revision_id;
              $existing_entity->save();

              $uuid_to_entity_map[$uuid] = $existing_entity;
            }
            else {
              // Create new entity
              $this->logger->info('Creating new menu link for UUID @uuid', ['@uuid' => $uuid]);

              $values = [];
              foreach ($menu_link_revision->getFields(FALSE) as $field_name => $field) {
                if (!in_array($field_name, ['revision_id', 'revision_created', 'revision_user', 'revision_log', 'changed', 'id', 'uuid', 'parent'])) {
                  $values[$field_name] = $field->getValue();
                }
              }

              $new_entity = $menu_link_storage->create($values);
              $new_entity->set('uuid', $uuid);  // Critical to maintain plugin IDs
              $new_entity->set('weight', $item_data['weight']);
              $new_entity->set('enabled', TRUE);
              $new_entity->set('parent', '');  // Temporarily clear parent

              $new_entity->enforceIsNew();
              $new_entity->revision_log = 'Created as part of menu revision revert ' . $menu_revision_id;
              $new_entity->save();

              $uuid_to_entity_map[$uuid] = $new_entity;
            }
          }
          catch (\Exception $e) {
            $this->logger->error('Error processing menu link UUID @uuid: @message', [
              '@uuid' => $uuid,
              '@message' => $e->getMessage()
            ]);
          }
        }
      }

      // Second pass: Set parent relationships now that all entities exist
      foreach ($hierarchy_data as $uuid => $item_data) {
        if (!isset($uuid_to_entity_map[$uuid])) {
          continue;
        }

        $entity = $uuid_to_entity_map[$uuid];
        $parent_plugin_id = $item_data['parent'];

        if (!empty($parent_plugin_id)) {
          $this->logger->info('Setting parent @parent for menu link @id', [
            '@id' => $entity->id(),
            '@parent' => $parent_plugin_id
          ]);

          try {
            $entity->set('parent', $parent_plugin_id);
            $entity->save();
          }
          catch (\Exception $e) {
            $this->logger->error('Failed to set parent for menu link @id: @message', [
              '@id' => $entity->id(),
              '@message' => $e->getMessage()
            ]);
          }
        }
      }

      // Find items that exist now but not in the revision (disable them)
      $current_uuids = array_keys($current_links_by_uuid);
      $to_disable_uuids = array_diff($current_uuids, $revision_uuids);

      foreach ($to_disable_uuids as $uuid) {
        $entity = $current_links_by_uuid[$uuid];
        $this->logger->info('Disabling menu link @id (UUID: @uuid) not in revision', [
          '@id' => $entity->id(),
          '@uuid' => $uuid
        ]);

        $entity->set('enabled', FALSE);
        $entity->setNewRevision(TRUE);
        $entity->revision_log = 'Disabled during menu revision revert to ' . $menu_revision_id;
        $entity->save();
      }

      // Force menu rebuilding
      \Drupal::service('plugin.manager.menu.link')->rebuild();
      \Drupal::service('router.builder')->rebuild();

      // Clear caches
      Cache::invalidateTags(['menu:' . $menu_name]);
      \Drupal::service('cache.menu')->deleteAll();
      \Drupal::service('cache.render')->invalidateAll();

      // Create a new revision to mark this revert
      $menu_revision_id = $this->createRevisionFromMenu($menu_name);
      $this->cleanDefaultStatusForItemsNotInMenuRevisionID($menu_revision_id);
      return $menu_revision_id;
    }
    catch (\Exception $e) {
      $transaction->rollBack();
      $this->logger->error('Failed to revert menu revision: @message', ['@message' => $e->getMessage()]);
      return FALSE;
    }
  }

  /**
   * Generate a menu from a revision.
   */
  public function getMenuTreeLinksFromMenuRevision($menuID,$menurevisionid): array
  {
    $menuLinkManager = \Drupal::service('plugin.manager.menu.link');
    //get the menu revision from the menu revision entity from menu revision id
    $query = $this->entityTypeManager->getStorage('menu_revision')->getQuery()
      ->accessCheck(TRUE)
      ->condition('id', $menurevisionid)
      ->condition('menu_name', $menuID);
    $menu_revision = $query->execute();

    if (!$menu_revision) {
      return [];
    }


    //get all the menu items for this menu revision
    $menuItemRevisions = $this->database->select('menu_revision_link', 'mrl')
      ->fields('mrl', ['menu_link_revision_id'])
      ->condition('menu_revision_id', $menurevisionid)
      ->execute()
      ->fetchCol();

    $menu_items = [];

    foreach ($menuItemRevisions as $revision_id) {
      $menu_link_revision = $this->entityTypeManager
        ->getStorage('menu_link_content')
        ->loadRevision($revision_id);

      if ($menu_link_revision) {
        $pluginDefinition = $menu_link_revision->getPluginDefinition();

        $menu_link_object = $menuLinkManager->createInstance($menu_link_revision->getPluginId(), $pluginDefinition);

        //merge existing options with override revision id
        $options = $menu_link_object->getOptions();
        $options['overrideRevisionID'] = $revision_id;
        $overrideData = [
          'options' => $options
          ];
        //update options BUT DO NOT SAVE TO DB
        $menu_link_object->updateLink($overrideData,false);

        //$menu_link_object->overrideRevisionID = $revision_id;



        $menu_items[$menu_link_revision->uuid()] = $menu_link_object;

      }
    }

    return $menu_items;
  }

  /**
   * Generate a menu from a revision.
   */
  public function generateMenuFromRevision($menuID, $menurevisionid) {
    //get the menu revision from the menu revision entity from menu revision id
    $query = $this->entityTypeManager->getStorage('menu_revision')->getQuery()
      ->accessCheck(TRUE)
      ->condition('id', $menurevisionid)
      ->condition('menu_name', $menuID);
    $menu_revision = $query->execute();

    if (!$menu_revision) {
      return [];
    }


    //get all the menu items for this menu revision
    $menuItemRevisions = $this->database->select('menu_revision_link', 'mrl')
      ->fields('mrl', ['menu_link_revision_id'])
      ->condition('menu_revision_id', $menurevisionid)
      ->execute()
      ->fetchCol();

    $menu_items = [];

    foreach ($menuItemRevisions as $revision_id) {
      $menu_link_revision = $this->entityTypeManager
        ->getStorage('menu_link_content')
        ->loadRevision($revision_id);
      if ($menu_link_revision) {
        $menu_items[$menu_link_revision->uuid()] = $menu_link_revision;

      }
    }


    // Build a hierarchical menu tree array.
    // Build a hierarchical menu tree array compatible with $variables['items'].
    $menu_tree = [];
    $menu_items_by_uuid = [];
    foreach ($menu_items as $menu_item) {
      $uuid = $menu_item->uuid();
      $parent = $menu_item->getParentId();
      $item = [
        'menu_link_content_id' => $menu_item->id(),
        'title' => $menu_item->getTitle(),
        'url' => $menu_item->getUrlObject(),
        'below' => [],
        'attributes' => [],
        'original_link' => $menu_item,
        'plugin_id' => $uuid,
        'parent' => $parent,
        'localized_options' => [],
        'in_active_trail' => FALSE,
        'is_expanded' => FALSE,
        'is_collapsed' => FALSE,
        'is_active' => FALSE,
        'access' => TRUE,
        'weight' => $menu_item->getWeight(),
      ];
      $menu_items_by_uuid[$uuid] = $item;
    }

    // Build the tree structure.
    foreach ($menu_items_by_uuid as $uuid => &$item) {
      $parent = $item['parent'];
      if (!empty($parent) && isset($menu_items_by_uuid[$parent])) {
        $menu_items_by_uuid[$parent]['below'][] = &$item;
      } else {
        $menu_tree[$uuid] = &$item;
      }
    }
    unset($item);

    return $menu_tree;
  }

  public function getLatestActiveMenuRevision($menuID)
  {
    //get the menu revision from the menu revision entity from menu revision id
    $query = $this->entityTypeManager->getStorage('menu_revision')->getQuery()
      ->accessCheck(TRUE)
      ->condition('menu_name', $menuID)
      ->condition('status', '1')
      ->sort('id', 'DESC')
      ->range(0, 1);
    $menu_revision_ids = $query->execute();
    return reset($menu_revision_ids);
  }

  /**
   * {@inheritdoc}
   */
  public function publishDraftMenu($menu_name): bool {
    try {
      $default_revision = $this->getDefaultRevision($menu_name);
      if (!$default_revision) {
        $this->logger->error('No default menu revision found for @menu', ['@menu' => $menu_name]);
        return FALSE;
      }
      $default_revision->setStatus(1);
      $this->entityTypeManager->getStorage('menu_revision')->save($default_revision);
      \Drupal::service('plugin.manager.menu.link')->rebuild();
      \Drupal::service('router.builder')->rebuild();
      Cache::invalidateTags(['menu:' . $menu_name]);
      \Drupal::service('cache.menu')->deleteAll();
      \Drupal::service('cache.render')->invalidateAll();
      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to publish menu: @message', ['@message' => $e->getMessage()]);
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getLatestPublishedRevision($menu_name) {
    $revision_id = $this->getLatestActiveMenuRevision($menu_name);
    return $this->entityTypeManager->getStorage('menu_revision')->load($revision_id);
  }

  public function getLatestRevision($menu_name) {
    $query = $this->entityTypeManager->getStorage('menu_revision')->getQuery()
      ->accessCheck(TRUE)
      ->condition('menu_name', $menu_name)
      ->sort('created', 'DESC')
      ->range(0, 1);
    $ids = $query->execute();
    if (!$ids) {
      return NULL;
    }
    return reset($ids);
  }

  /**
   * check if the menu is revisionable, it is ONLY revisionable if its items are of type menu_link_content
   * @param $menu_id
   * @return void
   */
  public function isMenuRevisionable($menu_id)
  {
    $all_links = \Drupal::service('menu.link_tree')->load($menu_id, new MenuTreeParameters());
    foreach ($all_links as $link) {
      // Only check custom menu links (menu_link_content)

      if (strpos($link->link->getPluginId(),'menu_link_content:')===false) {
        return false;
      }
    }

    return true;
  }

  /**
   * Deletes the target Draft menu link in the menu_revision_link table.
   *
   * This will replace the menu link edit function.
   *
   * @param string $menu
   *   The menu machine name.
   * @param int $menu_link_content_id
   *   The menu link content ID.
   */
  public function deleteDraftMenuLinkRevision($menu, $menu_link_content_id): void {
    // Get the newest draft menu revision ID from menuID.
    $default_revision = $this->getDefaultRevision($menu)->id();

    $menu_link_storage = $this->entityTypeManager->getStorage('menu_link_content');
    $menu_link = $menu_link_storage->load($menu_link_content_id);
    if ($menu_link) {
      $menu_link->set('enabled', FALSE);
      $menu_link->save();
    }

    // Delete all entries in menu_revision_link for this menu item.
    $this->database->delete('menu_revision_link')
      ->condition('menu_revision_id', $default_revision)
      ->condition('menu_link_content_id', $menu_link_content_id)
      ->execute();
  }

  /**
   * Sets the default status to 0 for all menu revisions except the given one.
   *
   * @param int $menu_revision_id
   *   The menu revision ID to keep as default.
   */
  public function cleanDefaultStatusForItemsNotInMenuRevisionID($menu_revision_id): void {
    $this->database->update('menu_revision')
      ->fields(['is_default' => 0])
      ->condition('id', $menu_revision_id, '<>')
      ->execute();
  }

}
