<?php

namespace Drupal\Tests\menu_revisions\Unit\Controller;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityViewBuilderInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\PageCache\ResponsePolicy\KillSwitch;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Url;
use Drupal\menu_revisions\Controller\MenuRevisionAdvancedController;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use PHPUnit\Framework\TestCase;

/**
 * Unit tests for MenuRevisionAdvancedController.
 *
 * @group menu_revisions
 * @coversDefaultClass \Drupal\menu_revisions\Controller\MenuRevisionAdvancedController
 */
class MenuRevisionAdvancedControllerTest extends TestCase {

  /** @var \PHPUnit\Framework\MockObject\MockObject|Connection */
  protected $database;

  /** @var \PHPUnit\Framework\MockObject\MockObject|EntityTypeManagerInterface */
  protected $entityTypeManager;

  /** @var \PHPUnit\Framework\MockObject\MockObject|KillSwitch */
  protected $pageCacheKillSwitch;

  /** @var \PHPUnit\Framework\MockObject\MockObject|MessengerInterface */
  protected $messenger;

  /** @var \PHPUnit\Framework\MockObject\MockObject|TranslationInterface */
  protected $stringTranslation;

  /** @var \PHPUnit\Framework\MockObject\MockObject|RequestStack */
  protected $requestStack;

  /** @var \PHPUnit\Framework\MockObject\MockObject|Request */
  protected $request;

  /** @var \PHPUnit\Framework\MockObject\MockObject|UrlGeneratorInterface */
  protected $urlGenerator;

  /** @var object Test double for menu manager (stdClass with added methods) */
  protected $menuRevisionsManager;

  /** @var \Drupal\menu_revisions\Controller\MenuRevisionAdvancedController */
  protected $controller;

  protected function setUp(): void {
    parent::setUp();

    $this->database = $this->createMock(Connection::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->pageCacheKillSwitch = $this->createMock(KillSwitch::class);
    $this->messenger = $this->createMock(MessengerInterface::class);
    $this->stringTranslation = $this->createMock(TranslationInterface::class);
    $this->requestStack = $this->createMock(RequestStack::class);
    $this->request = $this->createMock(Request::class);
    $this->urlGenerator = $this->createMock(UrlGeneratorInterface::class);

    // stdClass test double for menu manager; only add the methods we need.
    $this->menuRevisionsManager = $this->getMockBuilder(\stdClass::class)
      ->addMethods(['deleteDraftMenuLinkRevision'])
      ->getMock();

    // Request stack returns our request object.
    $this->request->query = new ParameterBag();
    $this->requestStack->method('getCurrentRequest')->willReturn($this->request);

    // Make translate return a string if used; avoid strict behavior.
    $this->stringTranslation->method('translate')
      ->willReturnCallback(function ($string, $args = [], $options = []) {
        return is_array($args) && $args ? strtr($string, $args) : (string) $string;
      });

    $container = new ContainerBuilder();
    $container->set('database', $this->database);
    $container->set('entity_type.manager', $this->entityTypeManager);
    $container->set('page_cache_kill_switch', $this->pageCacheKillSwitch);
    $container->set('messenger', $this->messenger);
    $container->set('string_translation', $this->stringTranslation);
    $container->set('request_stack', $this->requestStack);
    $container->set('url_generator', $this->urlGenerator);
    $container->set('menu_revisions.manager', $this->menuRevisionsManager);
    \Drupal::setContainer($container);

    $this->controller = new MenuRevisionAdvancedController($this->database);
    $this->controller->setStringTranslation($this->stringTranslation);
  }

  public function testCreate() {
    $container = $this->createMock(ContainerInterface::class);
    $container->expects($this->once())
      ->method('get')
      ->with('database')
      ->willReturn($this->database);

    $controller = MenuRevisionAdvancedController::create($container);
    $this->assertInstanceOf(MenuRevisionAdvancedController::class, $controller);
  }

  public function testRevisionPreviewWithDefaultContent() {
    $this->pageCacheKillSwitch->expects($this->once())->method('trigger');

    $this->request->query->set('preview_nid', 'default');

    $nodeQuery = $this->createMock(QueryInterface::class);
    $nodeQuery->method('accessCheck')->with(TRUE)->willReturn($nodeQuery);
    $nodeQuery->method('execute')->willReturn([1, 2]);

    $nodeStorage = $this->createMock(EntityStorageInterface::class);
    $nodeStorage->method('getQuery')->willReturn($nodeQuery);

    $node1 = $this->createMock(NodeInterface::class);
    $node1->method('label')->willReturn('Node 1');

    $node2 = $this->createMock(NodeInterface::class);
    $node2->method('label')->willReturn('Node 2');

    $nodeStorage->method('loadMultiple')->with([1, 2])->willReturn([1 => $node1, 2 => $node2]);

    $this->entityTypeManager->method('getStorage')->with('node')->willReturn($nodeStorage);

    // Only assert that addStatus was called once; don't assert exact message shape.
    $this->messenger->expects($this->once())->method('addStatus');

    $result = $this->controller->revisionPreview('main', '123');

    $this->assertIsArray($result);
    $this->assertCount(2, $result);
    $this->assertArrayHasKey('#type', $result[0]);
    $this->assertEquals('container', $result[0]['#type']);
    $this->assertArrayHasKey('dropdown', $result[0]);
  }

  public function testRevisionPreviewWithSpecificNode() {
    $this->pageCacheKillSwitch->expects($this->once())->method('trigger');

    $this->request->query->set('preview_nid', '1');

    $nodeQuery = $this->createMock(QueryInterface::class);
    $nodeQuery->method('accessCheck')->with(TRUE)->willReturn($nodeQuery);
    $nodeQuery->method('execute')->willReturn([1, 2]);

    $nodeStorage = $this->createMock(EntityStorageInterface::class);
    $nodeStorage->method('getQuery')->willReturn($nodeQuery);

    $node1 = $this->createMock(NodeInterface::class);
    $node1->method('label')->willReturn('Node 1');

    $node2 = $this->createMock(NodeInterface::class);
    $node2->method('label')->willReturn('Node 2');

    $nodeStorage->method('loadMultiple')->with([1, 2])->willReturn([1 => $node1, 2 => $node2]);

    // For load(1) return the node; for any other id return null.
    $nodeStorage->method('load')->willReturnCallback(function ($id) use ($node1) {
      return $id === 1 ? $node1 : null;
    });

    $nodeViewBuilder = $this->createMock(EntityViewBuilderInterface::class);
    $nodeViewBuilder->method('view')->with($node1)->willReturn(['#cache' => ['max-age' => 0]]);

    $this->entityTypeManager->method('getStorage')->with('node')->willReturn($nodeStorage);
    $this->entityTypeManager->method('getViewBuilder')->with('node')->willReturn($nodeViewBuilder);

    $this->messenger->expects($this->once())->method('addStatus');

    $result = $this->controller->revisionPreview('main', '123');

    $this->assertIsArray($result);
    $this->assertCount(2, $result);
    $this->assertArrayHasKey('#type', $result[0]);
    $this->assertEquals('container', $result[0]['#type']);
    $this->assertArrayHasKey('dropdown', $result[0]);
    $this->assertArrayHasKey('node', $result[1]);
  }

  public function testRevisionPreviewWithInvalidNodeId() {
    $this->pageCacheKillSwitch->expects($this->once())->method('trigger');

    $this->request->query->set('preview_nid', '999');

    $nodeQuery = $this->createMock(QueryInterface::class);
    $nodeQuery->method('accessCheck')->with(TRUE)->willReturn($nodeQuery);
    $nodeQuery->method('execute')->willReturn([999]);

    $nodeStorage = $this->createMock(EntityStorageInterface::class);
    $nodeStorage->method('getQuery')->willReturn($nodeQuery);

    // loadMultiple returns empty array and load returns null for any id.
    $nodeStorage->method('loadMultiple')->with([999])->willReturn([]);
    $nodeStorage->method('load')->willReturn(null);

    $this->entityTypeManager->method('getStorage')->with('node')->willReturn($nodeStorage);

    // Ensure the view builder is not used.
    $this->entityTypeManager->expects($this->never())->method('getViewBuilder');

    $this->messenger->expects($this->once())->method('addStatus');

    $result = $this->controller->revisionPreview('main', '123');

    $this->assertIsArray($result);
    $this->assertCount(2, $result);
    $this->assertArrayHasKey('#type', $result[0]);
    $this->assertEquals('container', $result[0]['#type']);
    $this->assertArrayHasKey('dropdown', $result[0]);
    $this->assertArrayNotHasKey('node', $result[1]);
  }

  public function testRevisionPreviewWithNoNodes() {
    $this->pageCacheKillSwitch->expects($this->once())->method('trigger');

    $this->request->query->set('preview_nid', 'default');

    $nodeQuery = $this->createMock(QueryInterface::class);
    $nodeQuery->method('accessCheck')->with(TRUE)->willReturn($nodeQuery);
    $nodeQuery->method('execute')->willReturn([]);

    $nodeStorage = $this->createMock(EntityStorageInterface::class);
    $nodeStorage->method('getQuery')->willReturn($nodeQuery);
    $nodeStorage->method('loadMultiple')->with([])->willReturn([]);

    $this->entityTypeManager->method('getStorage')->with('node')->willReturn($nodeStorage);

    $this->messenger->expects($this->once())->method('addStatus');

    $result = $this->controller->revisionPreview('main', '123');

    $this->assertIsArray($result);
    $this->assertCount(2, $result);
    $this->assertArrayHasKey('#type', $result[0]);
    $this->assertEquals('container', $result[0]['#type']);
    $this->assertArrayHasKey('dropdown', $result[0]);

    $this->assertEquals(2, count($result[0]['dropdown']['#options']));
    $this->assertArrayHasKey('select', $result[0]['dropdown']['#options']);
    $this->assertArrayHasKey('default', $result[0]['dropdown']['#options']);
  }

  public function testDeleteDraftMenuItemRevision() {
    $menu = 'main';
    $menu_link_content_id = '42';

    $this->menuRevisionsManager->expects($this->once())
      ->method('deleteDraftMenuLinkRevision')
      ->with($menu, $menu_link_content_id);

    // Do not assert the exact message object; only assert the call occurred.
    $this->messenger->expects($this->once())->method('addStatus');

    $this->urlGenerator->expects($this->once())
      ->method('generateFromRoute')
      ->with('entity.menu.draft', ['menu' => $menu])
      ->willReturn('entity.menu.draft_path');

    $result = $this->controller->deleteDraftMenuItemRevision($menu, $menu_link_content_id);

    $this->assertNotNull($result);
  }

}
