<?php

namespace Drupal\Tests\menu_revisions\Unit\Controller;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\menu_revisions\Controller\MenuRevisionController;
use Drupal\menu_revisions\Services\MenuRevisionManagerInterface;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Controller\MenuRevisionController
 *
 * @group menu_revisions
 */
class MenuRevisionControllerTest extends UnitTestCase {

  /**
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * @var \Drupal\Core\Datetime\DateFormatterInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $dateFormatter;

  /**
   * @var \Drupal\menu_revisions\Services\MenuRevisionManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $menuRevisionManager;

  /**
   * @var \Drupal\Tests\menu_revisions\Unit\Controller\TestMenuRevisionController
   */
  protected $controller;

  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->dateFormatter = $this->createMock(DateFormatterInterface::class);
    $this->menuRevisionManager = $this->createMock(MenuRevisionManagerInterface::class);

    // Use subclass to expose protected methods.
    $this->controller = new TestMenuRevisionController(
      $this->entityTypeManager,
      $this->dateFormatter,
      $this->menuRevisionManager
    );

    // Stub translation service to avoid ContainerNotInitializedException.
    $this->controller->setStringTranslation($this->getStringTranslationStub());
  }

  /**
   * @covers ::create
   */
  public function testCreateFromContainer() {
    $container = $this->createMock(ContainerInterface::class);

    $container->method('get')
      ->willReturnMap([
        ['entity_type.manager', ContainerInterface::EXCEPTION_ON_INVALID_REFERENCE, $this->entityTypeManager],
        ['date.formatter', ContainerInterface::EXCEPTION_ON_INVALID_REFERENCE, $this->dateFormatter],
        ['menu_revisions.manager', ContainerInterface::EXCEPTION_ON_INVALID_REFERENCE, $this->menuRevisionManager],
      ]);

    $controller = TestMenuRevisionController::create($container);
    $this->assertInstanceOf(MenuRevisionController::class, $controller);
  }

  /**
   * @covers ::revisionList
   */
  public function testRevisionListMenuNotFound() {
    $menu_storage = $this->createMock(EntityStorageInterface::class);
    $menu_storage->method('load')->willReturn(NULL);

    $this->entityTypeManager->method('getStorage')
      ->with('menu')
      ->willReturn($menu_storage);

    $result = $this->controller->revisionList('non_existing_menu');
    $this->assertArrayHasKey('#markup', $result);
    $this->assertEquals('Menu not found.', (string) $result['#markup']);
  }

  /**
   * @covers ::revisionListTitle
   */
  public function testRevisionListTitleWithEntity() {
    $menu_entity = $this->getMockBuilder('Drupal\Core\Entity\EntityInterface')->getMock();
    $menu_entity->method('label')->willReturn('Test Menu');

    $menu_storage = $this->createMock(EntityStorageInterface::class);
    $menu_storage->method('load')->willReturn($menu_entity);

    $this->entityTypeManager->method('getStorage')
      ->with('menu')
      ->willReturn($menu_storage);

    $title = $this->controller->revisionListTitle('test_menu');
    $this->assertEquals(
      'Revisions for <em class="placeholder">Test Menu</em> menu',
      (string) $title
    );
  }

  /**
   * @covers ::revisionListTitle
   */
  public function testRevisionListTitleNoEntity() {
    $menu_storage = $this->createMock(EntityStorageInterface::class);
    $menu_storage->method('load')->willReturn(NULL);

    $this->entityTypeManager->method('getStorage')
      ->with('menu')
      ->willReturn($menu_storage);

    $title = $this->controller->revisionListTitle('missing_menu');
    $this->assertEquals('Menu revisions', (string) $title);
  }

  /**
   * @covers ::viewRevisionStructure
   */
  public function testViewRevisionStructureRevisionNotFound() {
    $revision_storage = $this->createMock(EntityStorageInterface::class);
    $revision_storage->method('load')->willReturn(NULL);

    $this->entityTypeManager->method('getStorage')
      ->with('menu_revision')
      ->willReturn($revision_storage);

    $result = $this->controller->viewRevisionStructure('menu', 123);
    $this->assertArrayHasKey('#markup', $result);
    $this->assertEquals('Menu revision not found.', (string) $result['#markup']);
  }

  /**
   * @covers ::buildMenuTree
   */
  public function testBuildMenuTreeEmpty() {
    $result = $this->controller->publicBuildMenuTree('', [], []);
    $this->assertEquals([], $result);
  }

}

/**
 * Subclass exposing protected methods for testing.
 */
class TestMenuRevisionController extends MenuRevisionController {
  public function publicBuildMenuTree($parent, array $items_by_parent, array $all_items) {
    return parent::buildMenuTree($parent, $items_by_parent, $all_items);
  }
}
