<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\menu_revisions\Entity\MenuRevision;
use Drupal\Tests\UnitTestCase;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Minimal fake field type class to satisfy BaseFieldDefinition static calls in unit tests.
 *
 * By defining it here (in the test file's namespace) PHP will autoload it for the test run.
 */
class FakeFieldType {
  public static function getDefaultStorageSettings(): array {
    return [];
  }

  public static function getDefaultFieldSettings(): array {
    return [];
  }

  public static function schema(): array {
    return [];
  }

  public static function propertyDefinitions($field_definition = NULL): array {
    return [];
  }
}

/**
 * @coversDefaultClass \Drupal\menu_revisions\Entity\MenuRevision
 *
 * Unit tests for the MenuRevision entity.
 */
class MenuRevisionTest extends UnitTestCase {

  /**
   * Set up a minimal container with required services so BaseFieldDefinition() works.
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    // Minimal translation service stub. translate() returns the original string,
    // formatPlural() returns singular or plural based on $count.
    $translator = $this->getMockBuilder(TranslationInterface::class)->getMock();
    $translator->method('translate')->willReturnCallback(function ($string, $args = [], $options = []) {
      return $string;
    });
    $translator->method('formatPlural')->willReturnCallback(function ($count, $singular, $plural, $args = [], $options = []) {
      return ($count == 1) ? $singular : $plural;
    });
    $container->set('string_translation', $translator);

    // Use the named FakeFieldType class defined above.
    $fake_field_type_class_name = FakeFieldType::class;

    // Minimal plugin manager stub to satisfy BaseFieldDefinition::create().
    $field_type_manager = new class ($fake_field_type_class_name) {
      private $class_name;
      public function __construct($class_name) {
        $this->class_name = $class_name;
      }
      public function hasDefinition($type) {
        // Return TRUE to indicate the field type exists for unit tests.
        return TRUE;
      }
      public function getDefinition($type) {
        // Return a definition array containing the class name that implements
        // the expected static methods. BaseFieldDefinition will call methods on
        // this class (statically), e.g. ::getDefaultStorageSettings().
        return [
          'id' => $type,
          'class' => $this->class_name,
        ];
      }
    };
    $container->set('plugin.manager.field.field_type', $field_type_manager);

    // Set the container. Use a real ContainerBuilder object (not NULL).
    \Drupal::setContainer($container);
  }

  /**
   * Clean up the container after each test.
   *
   * Replace with a fresh, empty ContainerBuilder() to avoid leaking services
   * between tests while satisfying the type requirement.
   */
  protected function tearDown(): void {
    \Drupal::setContainer(new ContainerBuilder());
    parent::tearDown();
  }

  /**
   * Test getters and setters for simple scalar fields.
   *
   * This test uses a partial mock of the MenuRevision entity and intercepts
   * get() / set() calls to verify that the entity methods call them with the
   * expected keys and that the returned values are interpreted correctly.
   *
   * @covers ::getCreatedTime
   * @covers ::setCreatedTime
   * @covers ::getMenuName
   * @covers ::setMenuName
   * @covers ::setOwner
   * @covers ::getOwner
   * @covers ::getOwnerId
   * @covers ::setOwnerId
   */
  public function testGettersAndSetters() {
    $storage = [];

    // Create a mock user to satisfy UserInterface typehint.
    $user = $this->getMockBuilder(UserInterface::class)->getMock();
    $user->method('id')->willReturn(42);

    // Partial mock of MenuRevision: intercept 'get', 'set', and 'getEntityType'.
    /** @var \PHPUnit\Framework\MockObject\MockObject|MenuRevision $entity */
    $entity = $this->getMockBuilder(MenuRevision::class)
      ->disableOriginalConstructor()
      ->setMethods(['get', 'set', 'getEntityType'])
      ->getMock();

    // Implement set() to store values in $storage and return $this (fluent).
    $entity->method('set')->willReturnCallback(function ($key, $value) use (&$storage, $entity) {
      $storage[$key] = $value;
      return $entity;
    });

    // Implement get() to return simple objects with ->value, or for 'uid' return
    // an object with ->entity and ->target_id.
    $entity->method('get')->willReturnCallback(function ($key) use (&$storage, $user) {
      $obj = new \stdClass();
      if ($key === 'uid') {
        $obj->entity = $user;
        $obj->target_id = isset($storage['uid']) ? $storage['uid'] : NULL;
      }
      else {
        $obj->value = isset($storage[$key]) ? $storage[$key] : NULL;
      }
      return $obj;
    });

    // Basic scalar setters/getters.
    $entity->setCreatedTime(1600000000);
    $this->assertEquals(1600000000, $entity->getCreatedTime());

    $entity->setMenuName('main-menu');
    $this->assertEquals('main-menu', $entity->getMenuName());

    $entity->setOwner($user);
    // getOwner returns ->entity.
    $this->assertSame($user, $entity->getOwner());
    $this->assertEquals(42, $entity->getOwnerId());

    $entity->setOwnerId(15);
    $this->assertEquals(15, $entity->getOwnerId());
  }

  /**
   * Test boolean setters/getters and edge cases.
   *
   * Covers setDefault(), isDefault(), setStatus(), getStatus(), isPublished(),
   * setPublished().
   *
   * @covers ::isDefault
   * @covers ::setDefault
   * @covers ::getStatus
   * @covers ::setStatus
   * @covers ::isPublished
   * @covers ::setPublished
   */
  public function testBooleanFlags() {
    $storage = [];

    /** @var \PHPUnit\Framework\MockObject\MockObject|MenuRevision $entity */
    $entity = $this->getMockBuilder(MenuRevision::class)
      ->disableOriginalConstructor()
      ->setMethods(['get', 'set', 'getEntityType'])
      ->getMock();

    $entity->method('set')->willReturnCallback(function ($key, $value) use (&$storage, $entity) {
      $storage[$key] = $value;
      return $entity;
    });

    $entity->method('get')->willReturnCallback(function ($key) use (&$storage) {
      $obj = new \stdClass();
      $obj->value = isset($storage[$key]) ? $storage[$key] : NULL;
      return $obj;
    });

    // setDefault must store 1 for truthy, 0 for falsey.
    $entity->setDefault('non-empty-string'); // truthy
    $this->assertTrue($entity->isDefault());
    $this->assertEquals(1, $storage['is_default']);

    $entity->setDefault(0); // falsey
    $this->assertFalse($entity->isDefault());
    $this->assertEquals(0, $storage['is_default']);

    // setStatus/getStatus/isPublished/setPublished
    $entity->setStatus(TRUE);
    $this->assertTrue($entity->getStatus());
    $this->assertTrue($entity->isPublished());

    $entity->setPublished(FALSE);
    $this->assertFalse($entity->getStatus());
    $this->assertFalse($entity->isPublished());
  }

  /**
   * Test setUnpublished() uses entity type's published key.
   *
   * Ensures setUnpublished reads getEntityType()->getKey('published') and
   * sets that field to FALSE.
   *
   * @covers ::setUnpublished
   */
  public function testSetUnpublished() {
    $storage = [];

    /** @var \PHPUnit\Framework\MockObject\MockObject|MenuRevision $entity */
    $entity = $this->getMockBuilder(MenuRevision::class)
      ->disableOriginalConstructor()
      ->setMethods(['get', 'set', 'getEntityType'])
      ->getMock();

    // set stores into storage and returns $this.
    $entity->method('set')->willReturnCallback(function ($key, $value) use (&$storage, $entity) {
      $storage[$key] = $value;
      return $entity;
    });

    // get simply wraps stored value into ->value.
    $entity->method('get')->willReturnCallback(function ($key) use (&$storage) {
      $obj = new \stdClass();
      $obj->value = isset($storage[$key]) ? $storage[$key] : NULL;
      return $obj;
    });

    // The entity type's published key is 'status' for our entity.
    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->method('getKey')->willReturnMap([['published', 'status']]);

    $entity->method('getEntityType')->willReturn($entity_type);

    // Start with published = TRUE.
    $entity->set('status', TRUE);
    $this->assertTrue((bool) $storage['status']);

    // Call setUnpublished, this should set 'status' => FALSE.
    $entity->setUnpublished();
    $this->assertArrayHasKey('status', $storage);
    $this->assertFalse($storage['status']);
  }
}
