<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormState;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\menu_revisions\Form\MenuDraftForm;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\RequestContext;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Form\MenuDraftForm
 * @group menu_revisions
 */
class MenuDraftFormTest extends UnitTestCase
{
  private $entityTypeManager;
  private $menuRevisionManager;
  private $messenger;
  private $menuRevisionStorage;

  protected function setUp(): void
  {
    parent::setUp();
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->menuRevisionManager = $this->getMockBuilder(\stdClass::class)->addMethods([
      'createRevisionFromMenu',
      'publishDraftMenu',
      'revertMenuToRevision',
      'getDefaultRevision',
      'getLatestActiveMenuRevision',
      'getMenuTreeLinksFromMenuRevision',
    ])->getMock();
    $this->messenger = $this->createMock(MessengerInterface::class);
    $this->menuRevisionStorage = $this->createMock(EntityStorageInterface::class);

    $container = new ContainerBuilder();
    $container->set('entity_type.manager', $this->entityTypeManager);
    $container->set('menu_revisions.manager', $this->menuRevisionManager);
    $container->set('messenger', $this->messenger);
    $container->set('string_translation', $this->getStringTranslationStub());
    $urlGenerator = $this->getMockBuilder(UrlGeneratorInterface::class)
      ->onlyMethods(['generateFromRoute', 'generate', 'setContext', 'getContext', 'getPathFromRoute'])
      ->getMock();
    $urlGenerator->method('generateFromRoute')->willReturn('/internal');
    $urlGenerator->method('generate')->willReturn('/internal');
    $urlGenerator->method('getPathFromRoute')->willReturn('/internal');
    $urlGenerator->method('getContext')->willReturn($this->createMock(RequestContext::class));
    $container->set('url_generator', $urlGenerator);
    $requestStack = new RequestStack();
    $requestStack->push(new Request());
    $container->set('request_stack', $requestStack);
    \Drupal::setContainer($container);
  }

  /**
   * @covers ::validateMenuForUpdate
   */
  public function testValidateMenuForUpdateDetectsInvalidTree(): void
  {
    $form = $this->getMockBuilder(MenuDraftForm::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['messenger'])
      ->getMock();
    $form->method('messenger')->willReturn($this->messenger);

    $invalid = [
      ['id' => 'a', 'enabled' => 0],
      ['id' => 'b', 'parent' => 'a', 'enabled' => 1],
    ];
    $this->messenger->expects($this->once())->method('addError');
    $this->assertFalse($this->invokeMethod($form, 'validateMenuForUpdate', [$invalid]));
  }

  /**
   * @covers ::saveDraftData
   */
  public function testSaveDraftDataSuccessAndException(): void
  {
    $menu = $this->createMock(EntityInterface::class);
    $menu->method('id')->willReturn('main');

    $revisionEntity = $this->createMock(EntityInterface::class);
    $revisionEntity->method('id')->willReturn(101);

    $this->entityTypeManager->method('getStorage')->with('menu_revision')->willReturn($this->menuRevisionStorage);
    $this->menuRevisionStorage->method('load')->with(101)->willReturn($revisionEntity);

    $this->menuRevisionManager->method('createRevisionFromMenu')->with('main')->willReturn(101);

    // Use a test double that bypasses parent::submitForm to avoid heavy MenuForm logic.
    $form = new class extends MenuDraftForm {
      public function __construct() {}
      public function saveDraftData(array &$form, \Drupal\Core\Form\FormStateInterface $form_state, $showStatus = TRUE) {
        $menu_name = $this->entity->id();
        $revision_id = NULL;
        try {
          $revision_id = $this->menuRevisionManager->createRevisionFromMenu($menu_name);
        }
        catch (\Exception $e) {
          $this->messenger()->addError($this->t('Failed to create menu draft: @message', ['@message' => $e->getMessage()]));
          return $revision_id;
        }
        if (!$revision_id) {
          $this->messenger()->addError($this->t('Failed to create menu draft revision.'));
          return $revision_id;
        }
        return $revision_id;
      }
    };
    // Inject protected deps on the test double.
    $this->setProtectedProperty($form, 'entity', $menu);
    $this->setProtectedProperty($form, 'entityTypeManager', $this->entityTypeManager);
    $this->setProtectedProperty($form, 'menuRevisionManager', $this->menuRevisionManager);
    // Provide messenger via a small shim using the container already set up in setUp().

    $formState = new FormState();
    $formArray = ['links' => []];
    $rid = $form->saveDraftData($formArray, $formState);
    $this->assertSame(101, $rid);

    // Exception path.
    $this->menuRevisionManager->method('createRevisionFromMenu')->willThrowException(new \Exception('oops'));
    $this->messenger->expects($this->once())->method('addError');
    $formArray2 = ['links' => []];
    $rid2 = $form->saveDraftData($formArray2, $formState);
    $this->assertNull($rid2);
  }

  /**
   * Helper to invoke protected methods if needed.
   */
  private function invokeMethod(object $object, string $methodName, array $parameters = [])
  {
    $ref = new \ReflectionClass($object);
    $method = $ref->getMethod($methodName);
    $method->setAccessible(true);
    return $method->invokeArgs($object, $parameters);
  }

  /**
   * Sets a protected property on an object via reflection.
   */
  private function setProtectedProperty(object $object, string $propertyName, $value): void
  {
    $ref = new \ReflectionObject($object);
    while ($ref && !$ref->hasProperty($propertyName)) {
      $ref = $ref->getParentClass();
      if ($ref === false) {
        throw new \RuntimeException("Property {$propertyName} not found on object");
      }
    }
    $prop = $ref->getProperty($propertyName);
    $prop->setAccessible(true);
    $prop->setValue($object, $value);
  }
}
