<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormState;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\menu_revisions\Form\MenuRevisionAdminForm;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Form\MenuRevisionAdminForm
 * @group menu_revisions
 */
class MenuRevisionAdminFormTest extends UnitTestCase
{
  /** @var \PHPUnit\Framework\MockObject\MockObject&ConfigFactoryInterface */
  protected $configFactory;

  /** @var \PHPUnit\Framework\MockObject\MockObject&EntityTypeManagerInterface */
  protected $entityTypeManager;

  /** @var \PHPUnit\Framework\MockObject\MockObject&EntityStorageInterface */
  protected $menuStorage;

  /** @var \PHPUnit\Framework\MockObject\MockObject&MessengerInterface */
  protected $messenger;

  /** @var \PHPUnit\Framework\MockObject\MockObject */
  protected $menuRevisionsManager;

  /** @var \PHPUnit\Framework\MockObject\MockObject&TypedConfigManagerInterface */
  protected $typedConfigManager;

  protected function setUp(): void
  {
    parent::setUp();

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->menuStorage = $this->createMock(EntityStorageInterface::class);
    $this->messenger = $this->createMock(MessengerInterface::class);
    $this->typedConfigManager = $this->createMock(TypedConfigManagerInterface::class);
    $this->menuRevisionsManager = $this->getMockBuilder(\stdClass::class)
      ->addMethods([
        'getLatestRevision',
        'createRevisionFromMenu',
        'publishDraftMenu',
        'isMenuRevisionable',
      ])->getMock();

    // Wire the container so that \Drupal::service() and other shortcuts work.
    $container = new ContainerBuilder();
    $container->set('config.factory', $this->configFactory);
    $container->set('entity_type.manager', $this->entityTypeManager);
    $container->set('messenger', $this->messenger);
    $container->set('config.typed', $this->typedConfigManager);
    $container->set('string_translation', $this->getStringTranslationStub());
    $container->set('menu_revisions.manager', $this->menuRevisionsManager);
    \Drupal::setContainer($container);
  }

  /**
   * @covers ::getFormId
   */
  public function testGetFormId(): void
  {
    $form = new MenuRevisionAdminForm($this->configFactory, $this->typedConfigManager);
    $this->assertSame('menu_revisions_admin_form', $form->getFormId());
  }

  /**
   * @covers ::getEditableConfigNames
   */
  public function testGetEditableConfigNames(): void
  {
    $form = new MenuRevisionAdminForm($this->configFactory, $this->typedConfigManager);
    $this->assertSame(['menu_revisions.settings'], $this->invokeMethod($form, 'getEditableConfigNames'));
  }

  /**
   * @covers ::buildForm
   */
  public function testBuildFormPopulatesMenuOptionsAndDefaults(): void
  {
    $menuA = $this->getMockBuilder(\stdClass::class)->addMethods(['id', 'label'])->getMock();
    $menuA->method('id')->willReturn('main');
    $menuA->method('label')->willReturn('Main');

    $menuB = $this->getMockBuilder(\stdClass::class)->addMethods(['id', 'label'])->getMock();
    $menuB->method('id')->willReturn('footer');
    $menuB->method('label')->willReturn('Footer');

    $this->entityTypeManager->method('getStorage')->with('menu')->willReturn($this->menuStorage);
    $this->menuStorage->method('loadMultiple')->willReturn(['main' => $menuA, 'footer' => $menuB]);

    $config = $this->createMock(Config::class);
    $config->method('get')->with('selected_menu')->willReturn(['main' => 'main']);
    $this->configFactory->method('getEditable')->with('menu_revisions.settings')->willReturn($config);

    $formObject = new MenuRevisionAdminForm($this->configFactory, $this->typedConfigManager);
    $formState = new FormState();
    $built = $formObject->buildForm([], $formState);

    $this->assertArrayHasKey('selected_menu', $built);
    $this->assertSame([
      'main' => 'Main',
      'footer' => 'Footer',
    ], $built['selected_menu']['#options']);
    $this->assertSame(['main' => 'main'], $built['selected_menu']['#default_value']);
  }

  /**
   * @covers ::submitForm
   */
  public function testSubmitFormSuccessCreatesDefaultRevisionAndSavesConfig(): void
  {
    // Selected menu values as checkboxes (truthy selected items remain).
    $selected = ['main' => 'main', 'footer' => 0];

    // Service behavior: no existing revision for 'main'.
    $this->menuRevisionsManager->expects($this->once())
      ->method('getLatestRevision')->with('main')->willReturn(null);
    $this->menuRevisionsManager->expects($this->once())
      ->method('createRevisionFromMenu')->with('main');
    $this->menuRevisionsManager->expects($this->once())
      ->method('publishDraftMenu')->with('main');
    $this->menuRevisionsManager->expects($this->once())
      ->method('isMenuRevisionable')->with('main')->willReturn(true);

    // Config save and reset.
    $editableConfig = $this->getMockBuilder(Config::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['set', 'save'])
      ->getMock();
    $editableConfig->expects($this->once())
      ->method('set')->with('selected_menu', $selected)->willReturnSelf();
    $editableConfig->expects($this->once())->method('save');

    $this->configFactory->method('getEditable')->with('menu_revisions.settings')->willReturn($editableConfig);
    // Ensure reset is invokable on the factory service.
    $this->configFactory->expects($this->once())
      ->method('reset')->with('menu_revisions.settings');

    // Build minimal form and submit.
    $formObject = new MenuRevisionAdminForm($this->configFactory, $this->typedConfigManager);
    $formState = new FormState();
    $formState->setValue('selected_menu', $selected);
    $form = [];
    $formObject->submitForm($form, $formState);
    $this->assertTrue(true); // If no exception, success.
  }

  /**
   * @covers ::submitForm
   */
  public function testSubmitFormStopsOnIncompatibleMenuAndShowsError(): void
  {
    $selected = ['main' => 'main'];

    $this->menuRevisionsManager->method('getLatestRevision')->with('main')->willReturn('rev-1');
    $this->menuRevisionsManager->method('isMenuRevisionable')->with('main')->willReturn(false);

    // Expect an error message and no config save.
    $this->messenger->expects($this->once())
      ->method('addError')
      ->with($this->callback(function ($markup) {
        // Accept either string or markup object; check string cast.
        return (string) $markup !== '';
      }));

    $this->configFactory->expects($this->never())->method('get');
    $this->configFactory->expects($this->never())->method('reset');

    $formObject = new MenuRevisionAdminForm($this->configFactory, $this->typedConfigManager);
    $formState = new FormState();
    $formState->setValue('selected_menu', $selected);
    $form = [];
    $formObject->submitForm($form, $formState);
  }

  /**
   * Helper to invoke protected methods if needed.
   */
  private function invokeMethod(object $object, string $methodName, array $parameters = [])
  {
    $ref = new \ReflectionClass($object);
    $method = $ref->getMethod($methodName);
    $method->setAccessible(true);
    return $method->invokeArgs($object, $parameters);
  }
}


