<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormState;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\RequestContext;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\menu_revisions\Form\MenuRevisionRevertForm;
use Drupal\menu_revisions\Services\MenuRevisionManagerInterface;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Form\MenuRevisionRevertForm
 * @group menu_revisions
 */
class MenuRevisionRevertFormTest extends UnitTestCase
{
  /** @var \PHPUnit\Framework\MockObject\MockObject&MenuRevisionManagerInterface */
  private $menuRevisionManager;

  /** @var \PHPUnit\Framework\MockObject\MockObject&EntityTypeManagerInterface */
  private $entityTypeManager;

  /** @var \PHPUnit\Framework\MockObject\MockObject&EntityStorageInterface */
  private $menuRevisionStorage;

  /** @var \PHPUnit\Framework\MockObject\MockObject&EntityStorageInterface */
  private $menuStorage;

  /** @var \PHPUnit\Framework\MockObject\MockObject&DateFormatterInterface */
  private $dateFormatter;

  /** @var \PHPUnit\Framework\MockObject\MockObject&MessengerInterface */
  private $messenger;

  protected function setUp(): void
  {
    parent::setUp();

    $this->menuRevisionManager = $this->createMock(MenuRevisionManagerInterface::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->menuRevisionStorage = $this->createMock(EntityStorageInterface::class);
    $this->menuStorage = $this->createMock(EntityStorageInterface::class);
    $this->dateFormatter = $this->createMock(DateFormatterInterface::class);
    $this->messenger = $this->createMock(MessengerInterface::class);

    $container = new ContainerBuilder();
    $container->set('menu_revisions.manager', $this->menuRevisionManager);
    $container->set('entity_type.manager', $this->entityTypeManager);
    $container->set('date.formatter', $this->dateFormatter);
    $container->set('messenger', $this->messenger);
    $container->set('string_translation', $this->getStringTranslationStub());
    $urlGenerator = $this->getMockBuilder(UrlGeneratorInterface::class)
      ->onlyMethods(['generateFromRoute', 'generate', 'setContext', 'getContext', 'getPathFromRoute'])
      ->getMock();
    $urlGenerator->method('generateFromRoute')->willReturn('/internal');
    $urlGenerator->method('generate')->willReturn('/internal');
    $urlGenerator->method('getPathFromRoute')->willReturn('/internal');
    $urlGenerator->method('getContext')->willReturn($this->createMock(RequestContext::class));
    $container->set('url_generator', $urlGenerator);
    $requestStack = new RequestStack();
    $requestStack->push(new Request());
    $container->set('request_stack', $requestStack);
    \Drupal::setContainer($container);
  }

  /**
   * @covers ::getFormId
   */
  public function testGetFormId(): void
  {
    $form = new MenuRevisionRevertForm($this->menuRevisionManager, $this->entityTypeManager, $this->dateFormatter);
    $this->assertSame('menu_revisions_revert_confirm', $form->getFormId());
  }

  /**
   * @covers ::buildForm
   */
  public function testBuildFormMissingRevisionAddsErrorAndRedirects(): void
  {
    $this->entityTypeManager->method('getStorage')->with('menu_revision')->willReturn($this->menuRevisionStorage);
    $this->menuRevisionStorage->method('load')->with(123)->willReturn(null);

    $this->messenger->expects($this->once())->method('addError');

    $form = new MenuRevisionRevertForm($this->menuRevisionManager, $this->entityTypeManager, $this->dateFormatter);
    $formState = new FormState();
    $result = $form->buildForm([], $formState, 'main', 123);

    $this->assertInstanceOf(RedirectResponse::class, $result);
  }

  /**
   * @covers ::getConfirmText
   * @covers ::getDescription
   * @covers ::getCancelUrl
   */
  public function testConfirmTextDescriptionAndCancelUrl(): void
  {
    $form = new MenuRevisionRevertForm($this->menuRevisionManager, $this->entityTypeManager, $this->dateFormatter);
    // Seed required context for cancel URL.
    $menuEntity = $this->getMockBuilder(\stdClass::class)->addMethods(['label'])->getMock();
    $menuEntity->method('label')->willReturn('Main');
    $this->entityTypeManager->method('getStorage')->willReturnMap([
      ['menu_revision', $this->menuRevisionStorage],
      ['menu', $this->menuStorage],
    ]);
    $menuRevisionMock = $this->getMockBuilder(\stdClass::class)->addMethods(['getCreatedTime', 'id'])->getMock();
    $menuRevisionMock->method('getCreatedTime')->willReturn(1700000000);
    $menuRevisionMock->method('id')->willReturn('rev-x');
    $this->menuRevisionStorage->method('load')->with(1)->willReturn($menuRevisionMock);
    $this->menuStorage->method('load')->with('main')->willReturn($menuEntity);
    $form->buildForm([], new FormState(), 'main', 1);
    $this->assertSame('Revert', (string) $form->getConfirmText());
    $this->assertNotEmpty((string) $form->getDescription());
    $this->assertSame('entity.menu_revisions.collection', $form->getCancelUrl()->getRouteName());
  }

  /**
   * @covers ::getQuestion
   */
  public function testGetQuestionUsesMenuLabelAndFormattedDate(): void
  {
    $menuRevision = $this->getMockBuilder(\stdClass::class)->addMethods(['getCreatedTime', 'id'])->getMock();
    $menuRevision->method('getCreatedTime')->willReturn(1700000000);
    $menuRevision->method('id')->willReturn('rev-1');

    $menuEntity = $this->getMockBuilder(\stdClass::class)->addMethods(['label'])->getMock();
    $menuEntity->method('label')->willReturn('Main Menu');

    $this->entityTypeManager->method('getStorage')->willReturnMap([
      ['menu_revision', $this->menuRevisionStorage],
      ['menu', $this->menuStorage],
    ]);
    $this->menuRevisionStorage->method('load')->with(42)->willReturn($menuRevision);
    $this->menuStorage->method('load')->with('main')->willReturn($menuEntity);
    $this->dateFormatter->method('format')->with(1700000000, 'short')->willReturn('11/14/2023');

    $form = new MenuRevisionRevertForm($this->menuRevisionManager, $this->entityTypeManager, $this->dateFormatter);
    $form->buildForm([], new FormState(), 'main', 42);
    $question = (string) $form->getQuestion();

    $this->assertStringContainsString('Main Menu', $question);
    $this->assertStringContainsString('11/14/2023', $question);
  }

  /**
   * @covers ::submitForm
   */
  public function testSubmitFormShowsStatusOnSuccessAndErrorOnFailure(): void
  {
    $menuRevision = $this->getMockBuilder(\stdClass::class)->addMethods(['getCreatedTime', 'id'])->getMock();
    $menuRevision->method('getCreatedTime')->willReturn(1700000000);
    $menuRevision->method('id')->willReturn('rev-2');
    $menuEntity = $this->getMockBuilder(\stdClass::class)->addMethods(['label'])->getMock();
    $menuEntity->method('label')->willReturn('Main');
    $this->entityTypeManager->method('getStorage')->willReturnMap([
      ['menu_revision', $this->menuRevisionStorage],
      ['menu', $this->menuStorage],
    ]);
    $this->menuRevisionStorage->method('load')->with(7)->willReturn($menuRevision);
    $this->menuStorage->method('load')->with('main')->willReturn($menuEntity);

    $form = new MenuRevisionRevertForm($this->menuRevisionManager, $this->entityTypeManager, $this->dateFormatter);
    $formState = new FormState();
    $form->buildForm([], $formState, 'main', 7);

    $this->menuRevisionManager
      ->method('revertMenuToRevision')
      ->with('rev-2')
      ->willReturnOnConsecutiveCalls(true, false);

    $this->messenger->expects($this->once())->method('addStatus');
    $formArray = [];
    $form->submitForm($formArray, $formState);

    $this->messenger->expects($this->once())->method('addError');
    $formArray2 = [];
    $newFormState = new FormState();
    // Seed required context again for second submit.
    $form->buildForm([], $newFormState, 'main', 7);
    $form->submitForm($formArray2, $newFormState);
  }
}


