<?php

namespace Drupal\Tests\menu_revisions\Unit\Manipulator;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Menu\MenuLinkInterface;
use Drupal\Core\Menu\MenuLinkTreeElement;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\menu_revisions\Manipulator\MenuLinkDraftManipulator;
use Drupal\menu_revisions\Services\MenuRevisionManagerInterface;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Manipulator\MenuLinkDraftManipulator
 * @group menu_revisions
 */
class MenuLinkDraftManipulatorTest extends UnitTestCase
{
  private $menuRevisionManager;
  private $routeMatch;
  private $configFactory;
  private $config;

  protected function setUp(): void
  {
    parent::setUp();

    // Create a mock that implements the interface and adds the extra method
    // that exists in the implementation but not in the interface.
    $this->menuRevisionManager = $this->getMockBuilder(MenuRevisionManagerInterface::class)
      ->onlyMethods([
        'createRevisionFromMenu',
        'captureMenuLinkRevisions',
        'getDefaultRevision',
        'revertMenuToRevision',
        'publishDraftMenu',
        'getLatestPublishedRevision',
      ])
      ->addMethods(['generateMenuFromRevision'])
      ->getMock();

    $this->routeMatch = $this->createMock(RouteMatchInterface::class);
    
    // Mock config factory and config.
    $this->config = $this->createMock(ImmutableConfig::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->configFactory->method('get')->with('menu_revisions.settings')->willReturn($this->config);
  }

  /**
   * @covers ::removeDeleteItemFromDraft
   * @covers ::isMenuRevisionEnabled
   */
  public function testRemoveDeleteItemFromDraftRemovesItemsNotInRevision(): void
  {
    $menuEntity = $this->getMockBuilder(\stdClass::class)->addMethods(['id'])->getMock();
    $menuEntity->method('id')->willReturn('main');
    $this->routeMatch->method('getParameter')->with('menu')->willReturn($menuEntity);

    // Menu revisions is enabled for this menu.
    $this->config->method('get')->with('selected_menu')->willReturn(['main']);

    $draftRevision = $this->getMockBuilder(\stdClass::class)->addMethods(['id'])->getMock();
    $draftRevision->method('id')->willReturn(42);
    $this->menuRevisionManager->method('getDefaultRevision')->with('main')->willReturn($draftRevision);

    // Revision only has uuid-a, so uuid-b and uuid-c should be removed.
    $revisionItems = [
      'uuid-a' => ['title' => 'Item A'],
    ];
    $this->menuRevisionManager->method('generateMenuFromRevision')->with('main', 42)->willReturn($revisionItems);

    // Create tree with 3 items.
    $linkA = $this->createMock(MenuLinkInterface::class);
    $linkA->method('getPluginId')->willReturn('menu_link_content:uuid-a');
    $linkB = $this->createMock(MenuLinkInterface::class);
    $linkB->method('getPluginId')->willReturn('menu_link_content:uuid-b');
    $linkC = $this->createMock(MenuLinkInterface::class);
    $linkC->method('getPluginId')->willReturn('menu_link_content:uuid-c');

    $elementA = new MenuLinkTreeElement($linkA, false, 1, false, []);
    $elementB = new MenuLinkTreeElement($linkB, false, 1, false, []);
    $elementC = new MenuLinkTreeElement($linkC, false, 1, false, []);

    $tree = [
      'menu_link_content:uuid-a' => $elementA,
      'menu_link_content:uuid-b' => $elementB,
      'menu_link_content:uuid-c' => $elementC,
    ];

    $manipulator = new MenuLinkDraftManipulator($this->menuRevisionManager, $this->routeMatch, $this->configFactory);
    $result = $manipulator->removeDeleteItemFromDraft($tree);

    // Only uuid-a should remain.
    $this->assertCount(1, $result);
    $this->assertArrayHasKey('menu_link_content:uuid-a', $result);
    $this->assertArrayNotHasKey('menu_link_content:uuid-b', $result);
    $this->assertArrayNotHasKey('menu_link_content:uuid-c', $result);
  }

  /**
   * @covers ::removeDeleteItemFromDraft
   * @covers ::isMenuRevisionEnabled
   */
  public function testRemoveDeleteItemFromDraftPreservesAllItemsInRevision(): void
  {
    $menuEntity = $this->getMockBuilder(\stdClass::class)->addMethods(['id'])->getMock();
    $menuEntity->method('id')->willReturn('main');
    $this->routeMatch->method('getParameter')->with('menu')->willReturn($menuEntity);

    // Menu revisions is enabled for this menu.
    $this->config->method('get')->with('selected_menu')->willReturn(['main']);

    $draftRevision = $this->getMockBuilder(\stdClass::class)->addMethods(['id'])->getMock();
    $draftRevision->method('id')->willReturn(42);
    $this->menuRevisionManager->method('getDefaultRevision')->with('main')->willReturn($draftRevision);

    // All items are in revision.
    $revisionItems = [
      'uuid-a' => ['title' => 'Item A'],
      'uuid-b' => ['title' => 'Item B'],
    ];
    $this->menuRevisionManager->method('generateMenuFromRevision')->with('main', 42)->willReturn($revisionItems);

    $linkA = $this->createMock(MenuLinkInterface::class);
    $linkA->method('getPluginId')->willReturn('menu_link_content:uuid-a');
    $linkB = $this->createMock(MenuLinkInterface::class);
    $linkB->method('getPluginId')->willReturn('menu_link_content:uuid-b');

    $elementA = new MenuLinkTreeElement($linkA, false, 1, false, []);
    $elementB = new MenuLinkTreeElement($linkB, false, 1, false, []);

    $tree = [
      'menu_link_content:uuid-a' => $elementA,
      'menu_link_content:uuid-b' => $elementB,
    ];

    $manipulator = new MenuLinkDraftManipulator($this->menuRevisionManager, $this->routeMatch, $this->configFactory);
    $result = $manipulator->removeDeleteItemFromDraft($tree);

    // Both should remain.
    $this->assertCount(2, $result);
    $this->assertArrayHasKey('menu_link_content:uuid-a', $result);
    $this->assertArrayHasKey('menu_link_content:uuid-b', $result);
  }

  /**
   * @covers ::removeDeleteItemFromDraft
   * @covers ::isMenuRevisionEnabled
   */
  public function testRemoveDeleteItemFromDraftHandlesSubtreesRecursively(): void
  {
    $menuEntity = $this->getMockBuilder(\stdClass::class)->addMethods(['id'])->getMock();
    $menuEntity->method('id')->willReturn('main');
    $this->routeMatch->method('getParameter')->with('menu')->willReturn($menuEntity);

    // Menu revisions is enabled for this menu.
    $this->config->method('get')->with('selected_menu')->willReturn(['main']);

    $draftRevision = $this->getMockBuilder(\stdClass::class)->addMethods(['id'])->getMock();
    $draftRevision->method('id')->willReturn(42);
    $this->menuRevisionManager->method('getDefaultRevision')->with('main')->willReturn($draftRevision);

    // Parent and one child are in revision, other child is not.
    $revisionItems = [
      'uuid-parent' => ['title' => 'Parent'],
      'uuid-child-a' => ['title' => 'Child A'],
    ];
    $this->menuRevisionManager->method('generateMenuFromRevision')->with('main', 42)->willReturn($revisionItems);

    $linkParent = $this->createMock(MenuLinkInterface::class);
    $linkParent->method('getPluginId')->willReturn('menu_link_content:uuid-parent');
    $linkChildA = $this->createMock(MenuLinkInterface::class);
    $linkChildA->method('getPluginId')->willReturn('menu_link_content:uuid-child-a');
    $linkChildB = $this->createMock(MenuLinkInterface::class);
    $linkChildB->method('getPluginId')->willReturn('menu_link_content:uuid-child-b');

    $elementChildA = new MenuLinkTreeElement($linkChildA, false, 2, false, []);
    $elementChildB = new MenuLinkTreeElement($linkChildB, false, 2, false, []);
    $elementParent = new MenuLinkTreeElement($linkParent, false, 1, false, [
      'menu_link_content:uuid-child-a' => $elementChildA,
      'menu_link_content:uuid-child-b' => $elementChildB,
    ]);

    $tree = [
      'menu_link_content:uuid-parent' => $elementParent,
    ];

    $manipulator = new MenuLinkDraftManipulator($this->menuRevisionManager, $this->routeMatch, $this->configFactory);
    $result = $manipulator->removeDeleteItemFromDraft($tree);

    // Parent should remain, child-a should remain, child-b should be removed.
    $this->assertCount(1, $result);
    $this->assertArrayHasKey('menu_link_content:uuid-parent', $result);
    $this->assertCount(1, $result['menu_link_content:uuid-parent']->subtree);
    $this->assertArrayHasKey('menu_link_content:uuid-child-a', $result['menu_link_content:uuid-parent']->subtree);
    $this->assertArrayNotHasKey('menu_link_content:uuid-child-b', $result['menu_link_content:uuid-parent']->subtree);
  }

  /**
   * @covers ::removeDeleteItemFromDraft
   */
  public function testRemoveDeleteItemFromDraftReturnsTreeWhenNoMenuParameter(): void
  {
    // When menu parameter is null, return tree as-is.
    $this->routeMatch->method('getParameter')->with('menu')->willReturn(NULL);

    $linkA = $this->createMock(MenuLinkInterface::class);
    $linkA->method('getPluginId')->willReturn('menu_link_content:uuid-a');
    $elementA = new MenuLinkTreeElement($linkA, false, 1, false, []);

    $tree = [
      'menu_link_content:uuid-a' => $elementA,
    ];

    $manipulator = new MenuLinkDraftManipulator($this->menuRevisionManager, $this->routeMatch, $this->configFactory);
    $result = $manipulator->removeDeleteItemFromDraft($tree);

    // Tree should be returned unchanged.
    $this->assertCount(1, $result);
    $this->assertArrayHasKey('menu_link_content:uuid-a', $result);
  }

  /**
   * @covers ::removeDeleteItemFromDraft
   * @covers ::isMenuRevisionEnabled
   * @covers ::filterDisabledItems
   */
  public function testRemoveDeleteItemFromDraftFiltersDisabledWhenNoDraftRevisionAndRevisionsEnabled(): void
  {
    $menuEntity = $this->getMockBuilder(\stdClass::class)->addMethods(['id'])->getMock();
    $menuEntity->method('id')->willReturn('main');
    $this->routeMatch->method('getParameter')->with('menu')->willReturn($menuEntity);

    // Menu revisions IS enabled for this menu.
    $this->config->method('get')->with('selected_menu')->willReturn(['main']);

    // When no draft revision exists, filter disabled items.
    $this->menuRevisionManager->method('getDefaultRevision')->with('main')->willReturn(NULL);

    $linkA = $this->createMock(MenuLinkInterface::class);
    $linkA->method('getPluginId')->willReturn('menu_link_content:uuid-a');
    $linkA->method('isEnabled')->willReturn(TRUE);
    
    $linkB = $this->createMock(MenuLinkInterface::class);
    $linkB->method('getPluginId')->willReturn('menu_link_content:uuid-b');
    $linkB->method('isEnabled')->willReturn(FALSE);

    $elementA = new MenuLinkTreeElement($linkA, false, 1, false, []);
    $elementB = new MenuLinkTreeElement($linkB, false, 1, false, []);

    $tree = [
      'menu_link_content:uuid-a' => $elementA,
      'menu_link_content:uuid-b' => $elementB,
    ];

    $manipulator = new MenuLinkDraftManipulator($this->menuRevisionManager, $this->routeMatch, $this->configFactory);
    $result = $manipulator->removeDeleteItemFromDraft($tree);

    // Only enabled item should remain (disabled item is filtered out).
    $this->assertCount(1, $result);
    $this->assertArrayHasKey('menu_link_content:uuid-a', $result);
    $this->assertArrayNotHasKey('menu_link_content:uuid-b', $result);
  }

  /**
   * @covers ::removeDeleteItemFromDraft
   * @covers ::isMenuRevisionEnabled
   */
  public function testRemoveDeleteItemFromDraftHandlesStringMenuId(): void
  {
    // Test with string menu ID instead of entity object.
    $this->routeMatch->method('getParameter')->with('menu')->willReturn('main');

    // Menu revisions is enabled for this menu.
    $this->config->method('get')->with('selected_menu')->willReturn(['main']);

    $draftRevision = $this->getMockBuilder(\stdClass::class)->addMethods(['id'])->getMock();
    $draftRevision->method('id')->willReturn(42);
    $this->menuRevisionManager->method('getDefaultRevision')->with('main')->willReturn($draftRevision);

    $revisionItems = [
      'uuid-a' => ['title' => 'Item A'],
    ];
    $this->menuRevisionManager->method('generateMenuFromRevision')->with('main', 42)->willReturn($revisionItems);

    $linkA = $this->createMock(MenuLinkInterface::class);
    $linkA->method('getPluginId')->willReturn('menu_link_content:uuid-a');
    $linkA->method('isEnabled')->willReturn(TRUE);
    $linkB = $this->createMock(MenuLinkInterface::class);
    $linkB->method('getPluginId')->willReturn('menu_link_content:uuid-b');
    $linkB->method('isEnabled')->willReturn(TRUE);

    $elementA = new MenuLinkTreeElement($linkA, false, 1, false, []);
    $elementB = new MenuLinkTreeElement($linkB, false, 1, false, []);

    $tree = [
      'menu_link_content:uuid-a' => $elementA,
      'menu_link_content:uuid-b' => $elementB,
    ];

    $manipulator = new MenuLinkDraftManipulator($this->menuRevisionManager, $this->routeMatch, $this->configFactory);
    $result = $manipulator->removeDeleteItemFromDraft($tree);

    // Only uuid-a should remain.
    $this->assertCount(1, $result);
    $this->assertArrayHasKey('menu_link_content:uuid-a', $result);
    $this->assertArrayNotHasKey('menu_link_content:uuid-b', $result);
  }

  /**
   * @covers ::removeDeleteItemFromDraft
   * @covers ::isMenuRevisionEnabled
   */
  public function testRemoveDeleteItemFromDraftPreservesDisabledItemsWhenRevisionsNotEnabled(): void
  {
    $menuEntity = $this->getMockBuilder(\stdClass::class)->addMethods(['id'])->getMock();
    $menuEntity->method('id')->willReturn('footer');
    $this->routeMatch->method('getParameter')->with('menu')->willReturn($menuEntity);

    // Menu revisions is NOT enabled for 'footer' menu (only 'main' is enabled).
    $this->config->method('get')->with('selected_menu')->willReturn(['main']);

    $linkA = $this->createMock(MenuLinkInterface::class);
    $linkA->method('getPluginId')->willReturn('menu_link_content:uuid-a');
    $linkA->method('isEnabled')->willReturn(TRUE);
    
    $linkB = $this->createMock(MenuLinkInterface::class);
    $linkB->method('getPluginId')->willReturn('menu_link_content:uuid-b');
    $linkB->method('isEnabled')->willReturn(FALSE);

    $elementA = new MenuLinkTreeElement($linkA, false, 1, false, []);
    $elementB = new MenuLinkTreeElement($linkB, false, 1, false, []);

    $tree = [
      'menu_link_content:uuid-a' => $elementA,
      'menu_link_content:uuid-b' => $elementB,
    ];

    $manipulator = new MenuLinkDraftManipulator($this->menuRevisionManager, $this->routeMatch, $this->configFactory);
    $result = $manipulator->removeDeleteItemFromDraft($tree);

    // BOTH items should remain (standard Drupal behavior when revisions NOT enabled).
    // Disabled item shows with unchecked enabled box.
    $this->assertCount(2, $result);
    $this->assertArrayHasKey('menu_link_content:uuid-a', $result);
    $this->assertArrayHasKey('menu_link_content:uuid-b', $result);
  }
}

