<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\Insert;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\menu_revisions\Services\MenuHierarchyManager;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Services\MenuHierarchyManager
 * @group menu_revisions
 */
class MenuHierarchyManagerCaptureTest extends UnitTestCase {

  protected $hierarchyManager;
  protected $entityTypeManager;
  protected $database;
  protected $menuLinkStorage;
  protected $entityQuery;

  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->database = $this->createMock(Connection::class);
    $this->menuLinkStorage = $this->createMock(EntityStorageInterface::class);
    $this->entityQuery = $this->createMock(QueryInterface::class);

    $logger = $this->createMock(LoggerChannelInterface::class);
    $loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $loggerFactory->method('get')->willReturn($logger);

    $this->entityTypeManager->method('getStorage')
      ->with('menu_link_content')
      ->willReturn($this->menuLinkStorage);

    $this->menuLinkStorage->method('getQuery')->willReturn($this->entityQuery);

    $this->hierarchyManager = new MenuHierarchyManager(
      $this->entityTypeManager,
      $this->database,
      $loggerFactory
    );
  }

  /**
   * @covers ::captureMenuHierarchy
   */
  public function testCaptureMenuHierarchy() {
    $menu_name = 'test_menu';
    $menu_revision_id = 55;
    $menu_link_ids = [1];

    $this->entityQuery->method('accessCheck')->willReturnSelf();
    $this->entityQuery->method('condition')->willReturnSelf();
    $this->entityQuery->method('execute')->willReturn($menu_link_ids);

    $link = $this->createMock(MenuLinkContent::class);
    $link->method('id')->willReturn(1);
    $link->method('uuid')->willReturn('uuid-1');
    $link->method('getParentId')->willReturn('');
    $link->method('getWeight')->willReturn(10);

    $this->menuLinkStorage->method('loadMultiple')->with($menu_link_ids)->willReturn([1 => $link]);

    $insertMock = $this->createMock(Insert::class);
    $insertMock->expects($this->once())->method('fields')->with($this->arrayHasKey('uuid'))->willReturnSelf();
    $insertMock->expects($this->once())->method('execute');

    $this->database->expects($this->once())
      ->method('insert')
      ->with('menu_revision_hierarchy')
      ->willReturn($insertMock);

    $this->hierarchyManager->captureMenuHierarchy($menu_name, $menu_revision_id);
  }

  /**
   * @covers ::captureMenuHierarchy
   */
  public function testCaptureMenuHierarchyWithNoLinks() {
    $menu_name = 'empty_menu';
    $menu_revision_id = 99;

    // Mock the entity query to return no link ids.
    $this->entityQuery->method('accessCheck')->willReturnSelf();
    $this->entityQuery->method('condition')->willReturnSelf();
    $this->entityQuery->method('execute')->willReturn([]);

    // Database insert should never be called.
    $this->database->expects($this->never())->method('insert');

    $this->hierarchyManager->captureMenuHierarchy($menu_name, $menu_revision_id);
  }

  /**
   * @covers ::captureMenuHierarchy
   */
  public function testCaptureMenuHierarchyWithMultipleLinks() {
    $menu_name = 'multi_menu';
    $menu_revision_id = 101;
    $menu_link_ids = [1, 2];

    $this->entityQuery->method('accessCheck')->willReturnSelf();
    $this->entityQuery->method('condition')->willReturnSelf();
    $this->entityQuery->method('execute')->willReturn($menu_link_ids);

    $link1 = $this->createMock(\Drupal\menu_link_content\Entity\MenuLinkContent::class);
    $link1->method('id')->willReturn(1);
    $link1->method('uuid')->willReturn('uuid-1');
    $link1->method('getParentId')->willReturn('');
    $link1->method('getWeight')->willReturn(0);

    $link2 = $this->createMock(\Drupal\menu_link_content\Entity\MenuLinkContent::class);
    $link2->method('id')->willReturn(2);
    $link2->method('uuid')->willReturn('uuid-2');
    $link2->method('getParentId')->willReturn('menu_link_content:uuid-1');
    $link2->method('getWeight')->willReturn(5);

    $this->menuLinkStorage->method('loadMultiple')->willReturn([
      1 => $link1,
      2 => $link2,
    ]);

    $insertMock = $this->createMock(\Drupal\Core\Database\Query\Insert::class);
    $insertMock->method('fields')->willReturnSelf();
    $insertMock->method('execute')->willReturnOnConsecutiveCalls(1, 2);

    $this->database->expects($this->exactly(2))
      ->method('insert')
      ->with('menu_revision_hierarchy')
      ->willReturn($insertMock);

    $this->hierarchyManager->captureMenuHierarchy($menu_name, $menu_revision_id);
  }

}
