<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\menu_revisions\Services\MenuHierarchyManager;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Services\MenuHierarchyManager
 * @group menu_revisions
 */
class MenuHierarchyManagerGetTest extends UnitTestCase {

  protected $hierarchyManager;
  protected $database;

  protected function setUp(): void {
    parent::setUp();

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->database = $this->createMock(Connection::class);

    $logger = $this->createMock(LoggerChannelInterface::class);
    $loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $loggerFactory->method('get')->willReturn($logger);

    $this->hierarchyManager = new MenuHierarchyManager(
      $entityTypeManager,
      $this->database,
      $loggerFactory
    );
  }

  /**
   * @covers ::getMenuHierarchy
   */
  public function testGetMenuHierarchy() {
    $menu_revision_id = 77;

    $selectMock = $this->createMock(SelectInterface::class);
    $selectMock->method('fields')->willReturnSelf();
    $selectMock->method('condition')->willReturnSelf();
    $selectMock->method('execute')->willReturn([
      (object) [
        'uuid' => 'uuid-1',
        'menu_link_content_id' => 1,
        'parent' => '',
        'weight' => 5,
        'plugin_id' => 'menu_link_content:uuid-1',
      ],
    ]);

    $this->database->method('select')
      ->with('menu_revision_hierarchy', 'mrh')
      ->willReturn($selectMock);

    $result = $this->hierarchyManager->getMenuHierarchy($menu_revision_id);

    $this->assertArrayHasKey('uuid-1', $result);
    $this->assertSame(1, $result['uuid-1']['id']);
    $this->assertSame('', $result['uuid-1']['parent']);
    $this->assertSame(5, $result['uuid-1']['weight']);
    $this->assertSame('menu_link_content:uuid-1', $result['uuid-1']['plugin_id']);
  }

  /**
   * @covers ::getMenuHierarchy
   */
  public function testGetMenuHierarchyWithNoRecords() {
    $menu_revision_id = 200;

    $selectMock = $this->createMock(\Drupal\Core\Database\Query\SelectInterface::class);
    $selectMock->method('fields')->willReturnSelf();
    $selectMock->method('condition')->willReturnSelf();
    $selectMock->method('execute')->willReturn([]);

    $this->database->method('select')->willReturn($selectMock);

    $result = $this->hierarchyManager->getMenuHierarchy($menu_revision_id);
    $this->assertSame([], $result);
  }

  /**
   * @covers ::getMenuHierarchy
   */
  public function testGetMenuHierarchyWithMultipleRecords() {
    $menu_revision_id = 201;

    $selectMock = $this->createMock(\Drupal\Core\Database\Query\SelectInterface::class);
    $selectMock->method('fields')->willReturnSelf();
    $selectMock->method('condition')->willReturnSelf();
    $selectMock->method('execute')->willReturn([
      (object) [
        'uuid' => 'uuid-1',
        'menu_link_content_id' => 1,
        'parent' => '',
        'weight' => 0,
        'plugin_id' => 'menu_link_content:uuid-1',
      ],
      (object) [
        'uuid' => 'uuid-2',
        'menu_link_content_id' => 2,
        'parent' => 'menu_link_content:uuid-1',
        'weight' => 5,
        'plugin_id' => 'menu_link_content:uuid-2',
      ],
    ]);

    $this->database->method('select')->willReturn($selectMock);

    $result = $this->hierarchyManager->getMenuHierarchy($menu_revision_id);

    $this->assertCount(2, $result);
    $this->assertSame(1, $result['uuid-1']['id']);
    $this->assertSame('', $result['uuid-1']['parent']);
    $this->assertSame(0, $result['uuid-1']['weight']);
    $this->assertSame('menu_link_content:uuid-1', $result['uuid-1']['plugin_id']);

    $this->assertSame(2, $result['uuid-2']['id']);
    $this->assertSame('menu_link_content:uuid-1', $result['uuid-2']['parent']);
    $this->assertSame(5, $result['uuid-2']['weight']);
    $this->assertSame('menu_link_content:uuid-2', $result['uuid-2']['plugin_id']);
  }

}
