<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\Insert;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\menu_revisions\Services\MenuHierarchyManager;
use Drupal\menu_revisions\Services\MenuRevisionManager;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Services\MenuRevisionManager
 * @group menu_revisions
 */
class MenuRevisionManagerCaptureTest extends UnitTestCase {

  /**
   * The menu revision manager service.
   *
   * @var \Drupal\menu_revisions\Services\MenuRevisionManager
   */
  protected $menuRevisionManager;

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The mocked current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $currentUser;

  /**
   * The mocked database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $database;

  /**
   * The mocked logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * The mocked hierarchy manager.
   *
   * @var \Drupal\menu_revisions\Services\MenuHierarchyManager|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $hierarchyManager;

  /**
   * The mocked menu link content storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $menuLinkStorage;

  /**
   * The mocked entity query.
   *
   * @var \Drupal\Core\Entity\Query\QueryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityQuery;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mocks for dependencies.
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->currentUser = $this->createMock(AccountProxyInterface::class);
    $this->database = $this->createMock(Connection::class);
    $this->logger = $this->createMock(LoggerChannelInterface::class);
    $this->hierarchyManager = $this->createMock(MenuHierarchyManager::class);
    $this->menuLinkStorage = $this->createMock(EntityStorageInterface::class);
    $this->entityQuery = $this->createMock(QueryInterface::class);

    $logger_factory = $this->createMock(LoggerChannelFactoryInterface::class);
    $logger_factory->method('get')->willReturn($this->logger);

    // Set up the menu link storage mock.
    $this->entityTypeManager->method('getStorage')
      ->with('menu_link_content')
      ->willReturn($this->menuLinkStorage);

    // Configure the entity query mock.
    $this->menuLinkStorage->method('getQuery')
      ->willReturn($this->entityQuery);

    // Create the menu revision manager instance.
    $this->menuRevisionManager = new MenuRevisionManager(
      $this->entityTypeManager,
      $this->currentUser,
      $this->database,
      $logger_factory,
      $this->hierarchyManager
    );
  }

  /**
   * Tests capturing menu link revisions when links exist.
   *
   * @covers ::captureMenuLinkRevisions
   */
  public function testCaptureMenuLinkRevisionsWithLinks() {
    $menu_name = 'test_menu';
    $menu_revision_id = 123;
    $menu_link_ids = [1, 2, 3];

    // Configure entity query to return some menu link IDs.
    $this->entityQuery->expects($this->once())
      ->method('accessCheck')
      ->with(TRUE)
      ->willReturnSelf();

    $this->entityQuery->expects($this->exactly(2))
      ->method('condition')
      ->withConsecutive(
        ['menu_name', $menu_name],
        ['enabled', TRUE]
      )
      ->willReturnSelf();

    $this->entityQuery->expects($this->once())
      ->method('execute')
      ->willReturn($menu_link_ids);

    // Create mock menu link entities.
    $menu_links = [];
    foreach ($menu_link_ids as $id) {
      $menu_link = $this->createMock(MenuLinkContent::class);
      $menu_link->method('id')->willReturn($id);
      $menu_link->method('getRevisionId')->willReturn($id * 10); // Simple way to make unique revision IDs
      $menu_link->method('uuid')->willReturn("uuid-$id");
      $menu_link->method('getParentId')->willReturn('');
      $menu_link->method('getWeight')->willReturn($id);

      $menu_links[$id] = $menu_link;
    }

    // Configure storage to return the mock menu links.
    $this->menuLinkStorage->expects($this->once())
      ->method('loadMultiple')
      ->with($menu_link_ids)
      ->willReturn($menu_links);

    // Configure database insert operations.
    $insert_mock = $this->createMock(Insert::class);
    $insert_mock->expects($this->exactly(count($menu_link_ids)))
      ->method('fields')
      ->willReturnSelf();
    $insert_mock->expects($this->exactly(count($menu_link_ids)))
      ->method('execute');

    $this->database->expects($this->exactly(count($menu_link_ids)))
      ->method('insert')
      ->with('menu_revision_link')
      ->willReturn($insert_mock);

    // Call the method under test.
    $this->menuRevisionManager->captureMenuLinkRevisions($menu_name, $menu_revision_id);
  }

  /**
   * Tests capturing menu link revisions when no links exist.
   *
   * @covers ::captureMenuLinkRevisions
   */
  public function testCaptureMenuLinkRevisionsWithNoLinks() {
    $menu_name = 'empty_menu';
    $menu_revision_id = 123;

    // Configure entity query to return empty results.
    $this->entityQuery->expects($this->once())
      ->method('accessCheck')
      ->with(TRUE)
      ->willReturnSelf();

    $this->entityQuery->expects($this->exactly(2))
      ->method('condition')
      ->withConsecutive(
        ['menu_name', $menu_name],
        ['enabled', TRUE]
      )
      ->willReturnSelf();

    $this->entityQuery->expects($this->once())
      ->method('execute')
      ->willReturn([]);

    // Storage should not be called to load entities.
    $this->menuLinkStorage->expects($this->never())
      ->method('loadMultiple');

    // Database should not be called for inserts.
    $this->database->expects($this->never())
      ->method('insert');

    // Call the method under test.
    $this->menuRevisionManager->captureMenuLinkRevisions($menu_name, $menu_revision_id);
  }

  /**
   * Tests capturing menu links with parent relationships.
   *
   * @covers ::captureMenuLinkRevisions
   */
  public function testCaptureMenuLinkRevisionsWithParents() {
    $menu_name = 'hierarchical_menu';
    $menu_revision_id = 123;
    $menu_link_ids = [10, 20, 30];

    // Configure entity query.
    $this->entityQuery->method('accessCheck')->willReturnSelf();
    $this->entityQuery->method('condition')->willReturnSelf();
    $this->entityQuery->method('execute')->willReturn($menu_link_ids);

    // Create mock menu link entities with parent relationships.
    $menu_links = [];

    // Parent link.
    $parent_link = $this->createMock(MenuLinkContent::class);
    $parent_link->method('id')->willReturn(10);
    $parent_link->method('getRevisionId')->willReturn(100);
    $parent_link->method('uuid')->willReturn('parent-uuid');
    $parent_link->method('getParentId')->willReturn('');
    $parent_link->method('getWeight')->willReturn(0);
    $menu_links[10] = $parent_link;

    // First child link.
    $child_link1 = $this->createMock(MenuLinkContent::class);
    $child_link1->method('id')->willReturn(20);
    $child_link1->method('getRevisionId')->willReturn(200);
    $child_link1->method('uuid')->willReturn('child1-uuid');
    $child_link1->method('getParentId')->willReturn('menu_link_content:10');
    $child_link1->method('getWeight')->willReturn(0);
    $menu_links[20] = $child_link1;

    // Second child link.
    $child_link2 = $this->createMock(MenuLinkContent::class);
    $child_link2->method('id')->willReturn(30);
    $child_link2->method('getRevisionId')->willReturn(300);
    $child_link2->method('uuid')->willReturn('child2-uuid');
    $child_link2->method('getParentId')->willReturn('menu_link_content:10');
    $child_link2->method('getWeight')->willReturn(1);
    $menu_links[30] = $child_link2;

    // Configure storage.
    $this->menuLinkStorage->method('loadMultiple')->willReturn($menu_links);

    // Configure database insert operations.
    $insert_mock = $this->createMock(Insert::class);
    $insert_mock->method('fields')->willReturnSelf();
    $insert_mock->method('execute');
    $this->database->method('insert')->willReturn($insert_mock);

    // Call the method under test and assert it returns null (ensures at least one assertion).
    $result = $this->menuRevisionManager->captureMenuLinkRevisions($menu_name, $menu_revision_id);
    $this->assertNull($result);
  }

  /**
   * Tests exception handling during database operations.
   *
   * @covers ::captureMenuLinkRevisions
   */
  public function testCaptureMenuLinkRevisionsWithDatabaseError() {
    $menu_name = 'test_menu';
    $menu_revision_id = 123;
    $menu_link_ids = [1];

    // Configure entity query.
    $this->entityQuery->method('accessCheck')->willReturnSelf();
    $this->entityQuery->method('condition')->willReturnSelf();
    $this->entityQuery->method('execute')->willReturn($menu_link_ids);

    // Create mock menu link entity.
    $menu_link = $this->createMock(MenuLinkContent::class);
    $menu_link->method('id')->willReturn(1);
    $menu_link->method('getRevisionId')->willReturn(10);
    $menu_link->method('uuid')->willReturn('uuid-1');
    $menu_link->method('getParentId')->willReturn('');
    $menu_link->method('getWeight')->willReturn(1);

    // Configure storage to return the mock menu link.
    $this->menuLinkStorage->method('loadMultiple')->willReturn([1 => $menu_link]);

    // Simulate a database error.
    $insert_mock = $this->createMock(Insert::class);
    $insert_mock->method('fields')->willReturnSelf();
    $insert_mock->method('execute')->willThrowException(new \Exception('Database error'));

    $this->database->method('insert')->willReturn($insert_mock);

    // Expect exception to be thrown.
    $this->expectException(\Exception::class);
    $this->expectExceptionMessage('Database error');

    // Call the method under test.
    $this->menuRevisionManager->captureMenuLinkRevisions($menu_name, $menu_revision_id);
  }
}
