<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Database\Query\Update;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\menu_revisions\Services\MenuHierarchyManager;
use Drupal\menu_revisions\Services\MenuRevisionManager;
use Drupal\Tests\UnitTestCase;
use Prophecy\Argument;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Services\MenuRevisionManager
 * @group menu_revisions
 */
class MenuRevisionManagerCleanDefaultStatusForItemsNotInMenuRevisionIDTest extends UnitTestCase {

  /**
   * The menu revision manager under test.
   *
   * @var \Drupal\menu_revisions\Services\MenuRevisionManager
   */
  protected $menuRevisionManager;

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\Prophecy\Prophecy\ProphecyInterface
   */
  protected $entityTypeManager;

  /**
   * The mocked current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface|\Prophecy\Prophecy\ProphecyInterface
   */
  protected $currentUser;

  /**
   * The mocked database connection.
   *
   * @var \Drupal\Core\Database\Connection|\Prophecy\Prophecy\ProphecyInterface
   */
  protected $database;

  /**
   * The mocked logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\Prophecy\Prophecy\ProphecyInterface
   */
  protected $loggerFactory;

  /**
   * The mocked logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\Prophecy\Prophecy\ProphecyInterface
   */
  protected $logger;

  /**
   * The mocked menu hierarchy manager.
   *
   * @var \Drupal\menu_revisions\Services\MenuHierarchyManager|\Prophecy\Prophecy\ProphecyInterface
   */
  protected $hierarchyManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mock objects for dependencies.
    $this->entityTypeManager = $this->prophesize(EntityTypeManagerInterface::class);
    $this->currentUser = $this->prophesize(AccountProxyInterface::class);
    $this->database = $this->prophesize(Connection::class);
    $this->logger = $this->prophesize(LoggerChannelInterface::class);
    $this->loggerFactory = $this->prophesize(LoggerChannelFactoryInterface::class);
    $this->hierarchyManager = $this->prophesize(MenuHierarchyManager::class);

    // Configure logger factory mock to return logger mock.
    $this->loggerFactory->get('menu_revision')
      ->willReturn($this->logger->reveal());

    // Create the service under test.
    $this->menuRevisionManager = new MenuRevisionManager(
      $this->entityTypeManager->reveal(),
      $this->currentUser->reveal(),
      $this->database->reveal(),
      $this->loggerFactory->reveal(),
      $this->hierarchyManager->reveal()
    );
  }

  /**
   * @covers ::cleanDefaultStatusForItemsNotInMenuRevisionID
   */
  public function testCleanDefaultStatusForItemsNotInMenuRevisionID() {
    // Create mocks for database query builder objects.
    $update_query = $this->prophesize(Update::class);
    $statement = $this->prophesize(StatementInterface::class);

    // Set up expectations for the database query chain.
    $this->database->update('menu_revision')
      ->willReturn($update_query->reveal())
      ->shouldBeCalledOnce();

    $update_query->fields(['is_default' => 0])
      ->willReturn($update_query->reveal())
      ->shouldBeCalledOnce();

    $update_query->condition('id', 42, '<>')
      ->willReturn($update_query->reveal())
      ->shouldBeCalledOnce();

    $update_query->execute()
      ->willReturn($statement->reveal())
      ->shouldBeCalledOnce();

    // Call the method being tested.
    $this->menuRevisionManager->cleanDefaultStatusForItemsNotInMenuRevisionID(42);
  }

  /**
   * @covers ::cleanDefaultStatusForItemsNotInMenuRevisionID
   */
  public function testCleanDefaultStatusWithDifferentMenuRevisionID() {
    // Create mocks for database query builder objects.
    $update_query = $this->prophesize(Update::class);
    $statement = $this->prophesize(StatementInterface::class);

    // Set up expectations for the database query chain with a different ID.
    $this->database->update('menu_revision')
      ->willReturn($update_query->reveal())
      ->shouldBeCalledOnce();

    $update_query->fields(['is_default' => 0])
      ->willReturn($update_query->reveal())
      ->shouldBeCalledOnce();

    $update_query->condition('id', 100, '<>')
      ->willReturn($update_query->reveal())
      ->shouldBeCalledOnce();

    $update_query->execute()
      ->willReturn($statement->reveal())
      ->shouldBeCalledOnce();

    // Call the method being tested with a different ID.
    $this->menuRevisionManager->cleanDefaultStatusForItemsNotInMenuRevisionID(100);
  }

  /**
   * @covers ::cleanDefaultStatusForItemsNotInMenuRevisionID
   */
  public function testCleanDefaultStatusHandlesDatabaseErrors() {
    // Create mocks for database query builder objects.
    $update_query = $this->prophesize(Update::class);

    // Set up the database update chain to throw an exception.
    $this->database->update('menu_revision')
      ->willReturn($update_query->reveal())
      ->shouldBeCalledOnce();

    $update_query->fields(['is_default' => 0])
      ->willReturn($update_query->reveal())
      ->shouldBeCalledOnce();

    $update_query->condition('id', 42, '<>')
      ->willReturn($update_query->reveal())
      ->shouldBeCalledOnce();

    $update_query->execute()
      ->willThrow(new \Exception('Database error'))
      ->shouldBeCalledOnce();

    // We expect the exception to bubble up as the method doesn't catch it.
    $this->expectException(\Exception::class);
    $this->expectExceptionMessage('Database error');

    // Call the method being tested.
    $this->menuRevisionManager->cleanDefaultStatusForItemsNotInMenuRevisionID(42);
  }
}
