<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Database\Transaction;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\menu_revisions\Entity\MenuRevisionInterface;
use Drupal\menu_revisions\Services\MenuHierarchyManager;
use Drupal\menu_revisions\Services\MenuRevisionManager;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Services\MenuRevisionManager
 * @group menu_revisions
 */
class MenuRevisionManagerCreateRevisionTest extends UnitTestCase {

  /**
   * The menu revision manager service.
   *
   * @var \Drupal\menu_revisions\Services\MenuRevisionManager
   */
  protected $menuRevisionManager;

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The mocked current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $currentUser;

  /**
   * The mocked database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $database;

  /**
   * The mocked logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * The mocked logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $loggerFactory;

  /**
   * The mocked hierarchy manager.
   *
   * @var \Drupal\menu_revisions\Services\MenuHierarchyManager|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $hierarchyManager;

  /**
   * The mocked entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $menuRevisionStorage;

  /**
   * The mocked database transaction.
   *
   * @var \Drupal\Core\Database\Transaction|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $transaction;

  /**
   * The mocked string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $stringTranslation;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mocks for dependencies.
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->currentUser = $this->createMock(AccountProxyInterface::class);
    $this->database = $this->createMock(Connection::class);
    $this->loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $this->logger = $this->createMock(LoggerChannelInterface::class);
    $this->hierarchyManager = $this->createMock(MenuHierarchyManager::class);
    $this->menuRevisionStorage = $this->createMock(EntityStorageInterface::class);
    $this->transaction = $this->createMock(Transaction::class);
    $this->stringTranslation = $this->createMock(TranslationInterface::class);

    // Set up basic expectations.
    $this->loggerFactory->expects($this->any())
      ->method('get')
      ->with('menu_revision')
      ->willReturn($this->logger);

    $this->entityTypeManager->expects($this->any())
      ->method('getStorage')
      ->with('menu_revision')
      ->willReturn($this->menuRevisionStorage);

    $this->database->expects($this->any())
      ->method('startTransaction')
      ->with('menu_revision_create')
      ->willReturn($this->transaction);

    // Create partial mock for menu revision manager to test createRevisionFromMenu.
    $this->menuRevisionManager = $this->getMockBuilder(MenuRevisionManager::class)
      ->setConstructorArgs([
        $this->entityTypeManager,
        $this->currentUser,
        $this->database,
        $this->loggerFactory,
        $this->hierarchyManager,
      ])
      ->onlyMethods(['getDefaultRevision', 'captureMenuLinkRevisions', 't'])
      ->getMock();

    // Set up string translation mock.
    $this->menuRevisionManager->expects($this->any())
      ->method('t')
      ->willReturnCallback(function ($string, $args = []) {
        return strtr($string, $args);
      });
  }

  /**
   * Tests successful creation of a menu revision when no default exists.
   *
   * @covers ::createRevisionFromMenu
   */
  public function testCreateRevisionFromMenuNoDefault() {
    $menu_name = 'test_menu';
    $revision_id = 123;
    $user_id = 1;

    // Mock the current user ID.
    $this->currentUser->expects($this->once())
      ->method('id')
      ->willReturn($user_id);

    // No default revision exists.
    $this->menuRevisionManager->expects($this->once())
      ->method('getDefaultRevision')
      ->with($menu_name)
      ->willReturn(NULL);

    // Mock the new menu revision entity.
    $menu_revision = $this->createMock(MenuRevisionInterface::class);
    $menu_revision
      ->method('id')
      ->willReturn($revision_id);

    // Expect create() to be called with correct parameters.
    $this->menuRevisionStorage->expects($this->once())
      ->method('create')
      ->with($this->callback(function ($data) use ($menu_name, $user_id) {
        return $data['menu_name'] === $menu_name &&
          strpos($data['label'], 'Revision from') === 0 &&
          $data['uid'] === $user_id &&
          $data['is_default'] === TRUE &&
          $data['status'] === 0;
      }))
      ->willReturn($menu_revision);

    // Expect save() to be called.
    $this->menuRevisionStorage->expects($this->once())
      ->method('save')
      ->with($menu_revision);

    // Expect captureMenuLinkRevisions() to be called.
    $this->menuRevisionManager->expects($this->once())
      ->method('captureMenuLinkRevisions')
      ->with($menu_name, $revision_id);

    // Expect captureMenuHierarchy() to be called.
    $this->hierarchyManager->expects($this->once())
      ->method('captureMenuHierarchy')
      ->with($menu_name, $revision_id);

    // Call the method and assert result.
    $result = $this->menuRevisionManager->createRevisionFromMenu($menu_name);
    $this->assertEquals($revision_id, $result);
  }

  /**
   * Tests creation of a menu revision with an existing default.
   *
   * @covers ::createRevisionFromMenu
   */
  public function testCreateRevisionFromMenuWithExistingDefault() {
    $menu_name = 'test_menu';
    $revision_id = 123;
    $user_id = 1;

    // Mock the current user ID.
    $this->currentUser->expects($this->once())
      ->method('id')
      ->willReturn($user_id);

    // Mock existing default revision.
    $existing_revision = $this->createMock(MenuRevisionInterface::class);
    $existing_revision->expects($this->once())
      ->method('setDefault')
      ->with(FALSE);

    $this->menuRevisionManager->expects($this->once())
      ->method('getDefaultRevision')
      ->with($menu_name)
      ->willReturn($existing_revision);

    // Mock the new menu revision entity.
    $menu_revision = $this->createMock(MenuRevisionInterface::class);
    $menu_revision
      ->method('id')
      ->willReturn($revision_id);

    // Expect save() to be called twice:
    //  - first with the existing revision (to unset default)
    //  - second with the new revision (to persist it)
    $this->menuRevisionStorage->expects($this->exactly(2))
      ->method('save')
      ->withConsecutive(
        [$existing_revision],
        [$menu_revision]
      );

    // Expect create() to be called and return the new revision.
    $this->menuRevisionStorage->expects($this->once())
      ->method('create')
      ->willReturn($menu_revision);

    // Call the method and assert result.
    $result = $this->menuRevisionManager->createRevisionFromMenu($menu_name);
    $this->assertEquals($revision_id, $result);
  }

  /**
   * Tests creating a published revision.
   *
   * @covers ::createRevisionFromMenu
   */
  public function testCreatePublishedRevision() {
    $menu_name = 'test_menu';
    $revision_id = 123;
    $status = 1;

    // Mock the current user ID.
    $this->currentUser->expects($this->once())
      ->method('id')
      ->willReturn(1);

    // No default revision exists.
    $this->menuRevisionManager->expects($this->once())
      ->method('getDefaultRevision')
      ->willReturn(NULL);

    // Mock the new menu revision entity.
    $menu_revision = $this->createMock(MenuRevisionInterface::class);
    $menu_revision
      ->method('id')
      ->willReturn($revision_id);

    // Expect create() to be called with status = 1.
    $this->menuRevisionStorage->expects($this->once())
      ->method('create')
      ->with($this->callback(function ($data) use ($status) {
        return $data['status'] === $status;
      }))
      ->willReturn($menu_revision);

    // Call the method with status = 1.
    $result = $this->menuRevisionManager->createRevisionFromMenu($menu_name, $status);
    $this->assertEquals($revision_id, $result);
  }

  /**
   * Tests exception handling when creating a menu revision.
   *
   * @covers ::createRevisionFromMenu
   */
  public function testCreateRevisionFromMenuException() {
    $menu_name = 'test_menu';
    $exception_message = 'Storage failure';

    // Mock the current user ID.
    $this->currentUser->expects($this->once())
      ->method('id')
      ->willReturn(1);

    // No default revision exists.
    $this->menuRevisionManager->expects($this->once())
      ->method('getDefaultRevision')
      ->willReturn(NULL);

    // Simulate exception in create().
    $this->menuRevisionStorage->expects($this->once())
      ->method('create')
      ->willThrowException(new \Exception($exception_message));

    // Expect transaction rollback.
    $this->transaction->expects($this->once())
      ->method('rollBack');

    // Expect error to be logged.
    $this->logger->expects($this->once())
      ->method('error')
      ->with('Failed to create menu revision: @message', ['@message' => $exception_message]);

    // Call the method and expect exception.
    $this->expectException(\Exception::class);
    $this->expectExceptionMessage($exception_message);
    $this->menuRevisionManager->createRevisionFromMenu($menu_name);
  }

  /**
   * Tests exception handling when capturing menu link revisions.
   *
   * @covers ::createRevisionFromMenu
   */
  public function testCaptureMenuLinkRevisionsException() {
    $menu_name = 'test_menu';
    $revision_id = 123;
    $exception_message = 'Capture failure';

    // Mock the current user ID.
    $this->currentUser->expects($this->once())
      ->method('id')
      ->willReturn(1);

    // No default revision exists.
    $this->menuRevisionManager->expects($this->once())
      ->method('getDefaultRevision')
      ->willReturn(NULL);

    // Mock the new menu revision entity.
    $menu_revision = $this->createMock(MenuRevisionInterface::class);
    $menu_revision->expects($this->once())
      ->method('id')
      ->willReturn($revision_id);

    // Setup storage create and save.
    $this->menuRevisionStorage->expects($this->once())
      ->method('create')
      ->willReturn($menu_revision);
    $this->menuRevisionStorage->expects($this->once())
      ->method('save')
      ->with($menu_revision);

    // Simulate exception in captureMenuLinkRevisions().
    $this->menuRevisionManager->expects($this->once())
      ->method('captureMenuLinkRevisions')
      ->with($menu_name, $revision_id)
      ->willThrowException(new \Exception($exception_message));

    // Expect transaction rollback.
    $this->transaction->expects($this->once())
      ->method('rollBack');

    // Expect error to be logged.
    $this->logger->expects($this->once())
      ->method('error')
      ->with('Failed to create menu revision: @message', ['@message' => $exception_message]);

    // Call the method and expect exception.
    $this->expectException(\Exception::class);
    $this->expectExceptionMessage($exception_message);
    $this->menuRevisionManager->createRevisionFromMenu($menu_name);
  }

  /**
   * Tests exception handling when capturing menu hierarchy.
   *
   * @covers ::createRevisionFromMenu
   */
  public function testCaptureMenuHierarchyException() {
    $menu_name = 'test_menu';
    $revision_id = 123;
    $exception_message = 'Hierarchy capture failure';

    // Mock the current user ID.
    $this->currentUser->expects($this->once())
      ->method('id')
      ->willReturn(1);

    // No default revision exists.
    $this->menuRevisionManager->expects($this->once())
      ->method('getDefaultRevision')
      ->willReturn(NULL);

    // Mock the new menu revision entity.
    $menu_revision = $this->createMock(MenuRevisionInterface::class);
    $menu_revision->expects($this->any())
      ->method('id')
      ->willReturn($revision_id);

    // Setup storage create and save.
    $this->menuRevisionStorage->expects($this->once())
      ->method('create')
      ->willReturn($menu_revision);
    $this->menuRevisionStorage->expects($this->once())
      ->method('save')
      ->with($menu_revision);

    // Setup successful captureMenuLinkRevisions.
    $this->menuRevisionManager->expects($this->once())
      ->method('captureMenuLinkRevisions')
      ->with($menu_name, $revision_id);

    // Simulate exception in captureMenuHierarchy().
    $this->hierarchyManager->expects($this->once())
      ->method('captureMenuHierarchy')
      ->with($menu_name, $revision_id)
      ->willThrowException(new \Exception($exception_message));

    // Expect transaction rollback.
    $this->transaction->expects($this->once())
      ->method('rollBack');

    // Expect error to be logged.
    $this->logger->expects($this->once())
      ->method('error')
      ->with('Failed to create menu revision: @message', ['@message' => $exception_message]);

    // Call the method and expect exception.
    $this->expectException(\Exception::class);
    $this->expectExceptionMessage($exception_message);
    $this->menuRevisionManager->createRevisionFromMenu($menu_name);
  }

  /**
   * Tests with non-existent menu name.
   *
   * @covers ::createRevisionFromMenu
   */
  public function testNonExistentMenuName() {
    // This test verifies that the function handles a menu name that doesn't exist
    // Since the function itself doesn't validate menu existence, it should
    // proceed normally until something else (like captureMenuLinkRevisions) fails

    $menu_name = 'non_existent_menu';
    $revision_id = 123;
    $exception_message = 'Menu not found';

    // Mock the current user ID.
    $this->currentUser->expects($this->once())
      ->method('id')
      ->willReturn(1);

    // No default revision exists.
    $this->menuRevisionManager->expects($this->once())
      ->method('getDefaultRevision')
      ->willReturn(NULL);

    // Mock the new menu revision entity.
    $menu_revision = $this->createMock(MenuRevisionInterface::class);
    $menu_revision->expects($this->once())
      ->method('id')
      ->willReturn($revision_id);

    // Setup storage create and save.
    $this->menuRevisionStorage->expects($this->once())
      ->method('create')
      ->willReturn($menu_revision);
    $this->menuRevisionStorage->expects($this->once())
      ->method('save')
      ->with($menu_revision);

    // Simulate captureMenuLinkRevisions failing because menu doesn't exist.
    $this->menuRevisionManager->expects($this->once())
      ->method('captureMenuLinkRevisions')
      ->with($menu_name, $revision_id)
      ->willThrowException(new \Exception($exception_message));

    // Expect transaction rollback.
    $this->transaction->expects($this->once())
      ->method('rollBack');

    // Call the method and expect exception.
    $this->expectException(\Exception::class);
    $this->expectExceptionMessage($exception_message);
    $this->menuRevisionManager->createRevisionFromMenu($menu_name);
  }
}
