<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\menu_revisions\Entity\MenuRevisionInterface;
use Drupal\menu_revisions\Services\MenuHierarchyManager;
use Drupal\menu_revisions\Services\MenuRevisionManager;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Services\MenuRevisionManager
 * @group menu_revisions
 */
class MenuRevisionManagerDefaultRevisionTest extends UnitTestCase {

  /**
   * The menu revision manager service.
   *
   * @var \Drupal\menu_revisions\Services\MenuRevisionManager
   */
  protected $menuRevisionManager;

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The mocked entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $menuRevisionStorage;

  /**
   * The mocked query.
   *
   * @var \Drupal\Core\Entity\Query\QueryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $query;

  /**
   * The mocked current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $currentUser;

  /**
   * The mocked database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $database;

  /**
   * The mocked logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $loggerFactory;

  /**
   * The mocked logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * The mocked hierarchy manager.
   *
   * @var \Drupal\menu_revisions\Services\MenuHierarchyManager|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $hierarchyManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mocks for dependencies.
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->menuRevisionStorage = $this->createMock(EntityStorageInterface::class);
    $this->query = $this->createMock(QueryInterface::class);
    $this->currentUser = $this->createMock(AccountProxyInterface::class);
    $this->database = $this->createMock(Connection::class);
    $this->loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $this->logger = $this->createMock(LoggerChannelInterface::class);
    $this->hierarchyManager = $this->createMock(MenuHierarchyManager::class);

    // Set up basic expectations.
    $this->loggerFactory->expects($this->any())
      ->method('get')
      ->with('menu_revision')
      ->willReturn($this->logger);

    $this->entityTypeManager->expects($this->any())
      ->method('getStorage')
      ->with('menu_revision')
      ->willReturn($this->menuRevisionStorage);

    // Set up the service to test.
    $this->menuRevisionManager = new MenuRevisionManager(
      $this->entityTypeManager,
      $this->currentUser,
      $this->database,
      $this->loggerFactory,
      $this->hierarchyManager
    );
  }

  /**
   * Tests getting the default revision when it exists.
   *
   * @covers ::getDefaultRevision
   */
  public function testGetDefaultRevisionExists() {
    $menu_name = 'test_menu';
    $revision_id = 42;

    // Mock a query that will return a revision ID.
    $this->menuRevisionStorage->expects($this->once())
      ->method('getQuery')
      ->willReturn($this->query);

    $this->query->expects($this->once())
      ->method('accessCheck')
      ->with(TRUE)
      ->willReturn($this->query);

    $this->query->expects($this->exactly(2))
      ->method('condition')
      ->withConsecutive(
        ['menu_name', $menu_name],
        ['is_default', TRUE]
      )
      ->willReturn($this->query);

    $this->query->expects($this->once())
      ->method('execute')
      ->willReturn([$revision_id]);

    // Mock loading the revision.
    $expected_revision = $this->createMock(MenuRevisionInterface::class);
    $this->menuRevisionStorage->expects($this->once())
      ->method('load')
      ->with($revision_id)
      ->willReturn($expected_revision);

    // Call the method and assert the result.
    $result = $this->menuRevisionManager->getDefaultRevision($menu_name);
    $this->assertSame($expected_revision, $result);
  }

  /**
   * Tests getting the default revision when it doesn't exist.
   *
   * @covers ::getDefaultRevision
   */
  public function testGetDefaultRevisionNotExists() {
    $menu_name = 'test_menu';

    // Mock a query that will return empty results.
    $this->menuRevisionStorage->expects($this->once())
      ->method('getQuery')
      ->willReturn($this->query);

    $this->query->expects($this->once())
      ->method('accessCheck')
      ->with(TRUE)
      ->willReturn($this->query);

    $this->query->expects($this->exactly(2))
      ->method('condition')
      ->withConsecutive(
        ['menu_name', $menu_name],
        ['is_default', TRUE]
      )
      ->willReturn($this->query);

    $this->query->expects($this->once())
      ->method('execute')
      ->willReturn([]);

    // No load should be called.
    $this->menuRevisionStorage->expects($this->never())
      ->method('load');

    // Call the method and assert the result is null.
    $result = $this->menuRevisionManager->getDefaultRevision($menu_name);
    $this->assertNull($result);
  }
}
