<?php

namespace Drupal\Tests\menu_revisions\Unit;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\menu_revisions\Entity\MenuRevision;
use Drupal\menu_revisions\Services\MenuHierarchyManager;
use Drupal\menu_revisions\Services\MenuRevisionManager;
use Drupal\Tests\UnitTestCase;
use Prophecy\Argument;

/**
 * @coversDefaultClass \Drupal\menu_revisions\Services\MenuRevisionManager
 * @group menu_revisions
 */
class MenuRevisionManagerDeleteDraftMenuLinkRevisionTest extends UnitTestCase {

  /**
   * The menu revision manager under test.
   *
   * @var \Drupal\menu_revisions\Services\MenuRevisionManager
   */
  protected $menuRevisionManager;

  /**
   * The mocked entity type manager.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityTypeManager;

  /**
   * The mocked database connection.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $database;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->prophesize(EntityTypeManagerInterface::class);
    $this->database = $this->prophesize(Connection::class);
    
    // Set up other dependencies that aren't directly used in the test method
    $currentUser = $this->prophesize(AccountProxyInterface::class);
    $loggerFactory = $this->prophesize(LoggerChannelFactoryInterface::class);
    $logger = $this->prophesize(LoggerChannelInterface::class);
    $loggerFactory->get('menu_revision')->willReturn($logger->reveal());
    $hierarchyManager = $this->prophesize(MenuHierarchyManager::class);

    $this->menuRevisionManager = $this->getMockBuilder(MenuRevisionManager::class)
      ->setConstructorArgs([
        $this->entityTypeManager->reveal(),
        $currentUser->reveal(),
        $this->database->reveal(),
        $loggerFactory->reveal(),
        $hierarchyManager->reveal()
      ])
      ->setMethods(['getDefaultRevision'])
      ->getMock();
  }

  /**
   * @covers ::deleteDraftMenuLinkRevision
   */
  public function testDeleteDraftMenuLinkRevision() {
    $menu_name = 'main';
    $menu_link_content_id = 5;
    $revision_id = 10;
    
    // Mock default revision entity
    $menuRevision = $this->prophesize(MenuRevision::class);
    $menuRevision->id()->willReturn($revision_id);
    
    // Mock menu link entity
    $menuLinkContent = $this->prophesize(MenuLinkContent::class);
    
    // Mock storage interfaces
    $menuLinkStorage = $this->prophesize(EntityStorageInterface::class);
    $menuLinkStorage->load($menu_link_content_id)->willReturn($menuLinkContent->reveal());
    
    $this->entityTypeManager->getStorage('menu_link_content')
      ->willReturn($menuLinkStorage->reveal());
    
    // Set up mock for getDefaultRevision
    $this->menuRevisionManager->expects($this->once())
      ->method('getDefaultRevision')
      ->with($menu_name)
      ->willReturn($menuRevision->reveal());
    
    // Mock disabling the menu link
    $menuLinkContent->set('enabled', FALSE)->shouldBeCalled();
    $menuLinkContent->save()->shouldBeCalled();
    
    // Mock database delete operation
    $delete = $this->prophesize(\Drupal\Core\Database\Query\Delete::class);
    $delete->condition('menu_revision_id', $revision_id)->willReturn($delete->reveal());
    $delete->condition('menu_link_content_id', $menu_link_content_id)->willReturn($delete->reveal());
    $delete->execute()->shouldBeCalled();
    
    $this->database->delete('menu_revision_link')
      ->willReturn($delete->reveal());
    
    // Call the method under test
    $this->menuRevisionManager->deleteDraftMenuLinkRevision($menu_name, $menu_link_content_id);
    
    // The assertions are handled by the mock expectations
  }
}